#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include <varargs.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <machine/soundcard.h>
#include <errno.h>
#include <math.h>
#include <time.h>
#include <sys/time.h>
#include "ppsclock.h"

#define N_SOUNDFILES 66
#define LEN_PAUSE 0.36
#define LEN_PIP 0.7
#define START_SECOND 4

struct Snd { int	Length;
	     char	Data[ 1 ];
};

int PipLength = 22000;
int PauseLength = 16000;
int SoundSpeed = 8000;
int SoundBytes = 1;
int Debug = 0;
int UsePps = 0;
int DspFd = -1;
int PpsFd = -1;
char *FileNames[ N_SOUNDFILES ] = 
{
  "0.wav",
  "1.wav",
  "2.wav",
  "3.wav",
  "4.wav",
  "5.wav",
  "6.wav",
  "7.wav",
  "8.wav",
  "9.wav",
  "10.wav",
  "11.wav",
  "12.wav",
  "13.wav",
  "14.wav",
  "15.wav",
  "16.wav",
  "17.wav",
  "18.wav",
  "19.wav",
  "20.wav",
  "21.wav",
  "22.wav",
  "23.wav",
  "24.wav",
  "25.wav",
  "26.wav",
  "27.wav",
  "28.wav",
  "29.wav",
  "30.wav",
  "31.wav",
  "32.wav",
  "33.wav",
  "34.wav",
  "35.wav",
  "36.wav",
  "37.wav",
  "38.wav",
  "39.wav",
  "40.wav",
  "41.wav",
  "42.wav",
  "43.wav",
  "44.wav",
  "45.wav",
  "46.wav",
  "47.wav",
  "48.wav",
  "49.wav",
  "50.wav",
  "51.wav",
  "52.wav",
  "53.wav",
  "54.wav",
  "55.wav",
  "56.wav",
  "57.wav",
  "58.wav",
  "59.wav",
  "och0.wav",
  "och10.wav",
  "och20.wav",
  "och30.wav",
  "och40.wav",
  "och50.wav"
};

struct Snd *Sounds[ N_SOUNDFILES ];

char Pause[ 32000 ];
char Pip[ 64000 ];
char SineWave[ 256 ];

/*
 * Signal handler.
 */
void Terminate( int sig )
{

  if( Debug )
    printf( "Terminating\n" );

  if( DspFd >= 0 )
    close( DspFd );

  if( PpsFd >= 0 )
    close( PpsFd );

  exit( 0 );

}

/*
 * Disconnect the tty and enter daemon mode.
 */
void DisconnectFromTty()
{
  int i;

  if( fork() ) {
    exit(0);
  }
  i = getdtablesize();
  while (--i >= 0)
    (void)close(i);
  (void) open("/dev/null", O_RDONLY);
  (void) dup2(0, 1);
  (void) dup2(0, 2);
  i = open("/dev/tty", O_RDWR);
  if( i >= 0 ) {
    (void) ioctl(i, TIOCNOTTY, NULL);
    (void) close(i);
  }

}

/*
 * Wait until the last digit in the seconds are
 * equal to the desired value.
 */
void WaitUntil( int Seconds )
{

  while( ( time( NULL ) % 10 ) != Seconds )
    usleep( 100000 );

}

/*
 * Wait until the seconds shift.
 */
void WaitUntilZero()
{
  struct timeval tm;

  WaitUntil( 9 );
  if( UsePps ) {
    ioctl( PpsFd, CIOWAITPPS, 0 );
  }
  else {
    do {
      gettimeofday( &tm, NULL );
      if( tm.tv_usec < 970000L )
        usleep( 970000 - tm.tv_usec );
    }
    while( tm.tv_usec < 999990L );
  }

}

/*
 * Read all sound files and store them internally.
 * All files are in .wav format.
 * Sample rate and width is set in "sounds.conf".
 */
int LoadSounds( char *Directory )
{
  FILE *fp;
  int Digit, Length;
  char FileName[ 256 ];
  char Buf[ 256 ];
  struct Snd *Snds;

  sprintf( FileName, "%s/%s", Directory, "sounds.conf" );
  if( ( fp = fopen( FileName, "r" ) ) == NULL ) {
    printf( "Can't open sound config file %s\n", FileName );
    return( 1 );
  }
  if( ( Length = fread( Buf, 1, sizeof Buf - 1, fp ) ) <= 0 ) {
    printf( "Unexpected eof in %s\n", FileName );
    fclose( fp );
    return( 1 );
  }
  Buf[ Length ] = '\0';
  if( sscanf( Buf, "%d %d", &SoundSpeed, &SoundBytes ) != 2 ) {
    printf( "Syntax error in %s\n", FileName );
    fclose( fp );
    return( 1 );
  }
  fclose( fp );
  for( Digit = 0; Digit < N_SOUNDFILES; Digit++ ) {
    sprintf( FileName, "%s/%s", Directory, FileNames[ Digit ] );
    if( ( fp = fopen( FileName, "r" ) ) == NULL ) {
      printf( "Can't open sound file %s\n", FileName );
      return( 1 );
    }
    if( fread( Buf, 1, 4, fp ) != 4 ) {
      printf( "Unexpected eof in %s\n", FileName );
      fclose( fp );
      return( 1 );
    }
    if( memcmp( Buf, "RIFF", 4 ) != 0 ) {
      printf( "%s is not a WAVE file\n", FileName );
      fclose( fp );
      return( 1 );
    }
    if( fread( Buf, 1, sizeof( int ), fp ) != sizeof( int ) ) {
      printf( "Unexpected eof in %s\n", FileName );
      fclose( fp );
      return( 1 );
    }
    if( fread( Buf, 1, 4, fp ) != 4 ) {
      printf( "Unexpected eof in %s\n", FileName );
      fclose( fp );
      return( 1 );
    }
    if( memcmp( Buf, "WAVE", 4 ) != 0 ) {
      printf( "%s is not a WAVE file\n", FileName );
      fclose( fp );
      return( 1 );
    }
    if( fread( Buf, 1, 4, fp ) != 4 ) {
      printf( "Unexpected eof in %s\n", FileName );
      fclose( fp );
      return( 1 );
    }
    if( memcmp( Buf, "data", 4 ) != 0 ) {
      if( memcmp( Buf, "fmt ", 4 ) != 0 ) {
	printf( "%s is not a WAVE file\n", FileName );
	fclose( fp );
	return( 1 );
      }
      if( fread( Buf, 1, sizeof( int ), fp ) != sizeof( int ) ) {
	printf( "Unexpected eof in %s\n", FileName );
	fclose( fp );
	return( 1 );
      }
      Length = *((int *)(Buf));
      if( fread( Buf, 1, Length, fp ) != Length ) {
	printf( "Unexpected eof in %s\n", FileName );
	fclose( fp );
	return( 1 );
      }
      if( fread( Buf, 1, 4, fp ) != 4 ) {
	printf( "Unexpected eof in %s\n", FileName );
	fclose( fp );
	return( 1 );
      }
      if( memcmp( Buf, "data", 4 ) != 0 ) {
	printf( "Syntax error in %s\n", FileName );
	fclose( fp );
	return( 1 );
      }
    }
    if( fread( Buf, 1, sizeof( int ), fp ) != sizeof( int ) ) {
      printf( "Unexpected eof in %s\n", FileName );
      fclose( fp );
      return( 1 );
    }
    Length = *((int *)(Buf));
    if( ( Snds = malloc( Length + sizeof( int ) ) ) == NULL ) {
      printf( "Memory allocation error\n" );
      fclose( fp );
      return( 1 );
    }
    Snds->Length = Length;
    if( fread( Snds->Data, 1, Length, fp ) != Length ) {
      printf( "Unexpected eof in %s\n", FileName );
      fclose( fp );
      return( 1 );
    }
    Sounds[ Digit ] = Snds;
    fclose( fp );
  }
  return( 0 );

}

/*
 * Play a sound according to the supplied sound number.
 */
void PlaySound( int Sound )
{

  write( DspFd, Sounds[ Sound ]->Data, Sounds[ Sound ]->Length );

}

/*
 * Play 100 cycles of 1000 Hz.
 */
void PlayBeep()
{
  struct timeval Before, After;
  if( Debug > 1 )
    gettimeofday( &Before, NULL );

  write( DspFd, Pip, PipLength );
  if( Debug > 1 )
    gettimeofday( &After, NULL );

  ioctl( DspFd, SNDCTL_DSP_SYNC, 0 );
  if( Debug > 1 )
    printf( "us: before %ld after %ld\n", Before.tv_usec, After.tv_usec );

}

/*
 * Spell out the present time.
 */
void PlayTime()
{
  time_t Now;
  struct tm *Date;

  Now = time( NULL );
  Now += ( 10 - START_SECOND );
  Date = localtime( &Now );
  PlaySound( Date->tm_hour );
  write( DspFd, Pause, PauseLength );
  PlaySound( Date->tm_min );
  write( DspFd, Pause, PauseLength );
  PlaySound( ( Date->tm_sec / 10 ) + 60 );
  write( DspFd, Pause, PauseLength );
  ioctl( DspFd, SNDCTL_DSP_SYNC, 0 );

}

/*
 * Open and init sound device.
 */
int OpenSound( char *Device )
{
  int i;

  if( ( DspFd = open( Device, O_WRONLY ) ) < 0 ) {
    if( Debug )
      printf( "Can't open /dev/dsp, %s\n", strerror( errno ) );
    else
      syslog( LOG_ERR, "Can't open /dev/dsp, %m\n" );

    return( -1 );
  }
  i = ( SoundBytes > 1 ? AFMT_S16_LE : AFMT_U8 );
  if( ioctl( DspFd, SNDCTL_DSP_SETFMT, &i ) < 0 ) {
    if( Debug )
      printf( "Error received setting sound format\n" );
    else
      syslog( LOG_ERR, "Error received setting sound format" );

    close( DspFd );
    return( -1 );
  }
  i = 0;
  if( ioctl( DspFd, SNDCTL_DSP_STEREO, &i ) < 0 ) {
    if( Debug )
      printf( "Error received setting sound channels\n" );
    else
      syslog( LOG_ERR, "Error received setting sound channels" );

    close( DspFd );
    return( -1 );
  }
  i = SoundSpeed;
  if( ioctl( DspFd, SNDCTL_DSP_SPEED, &i ) < 0 ) {
    if( Debug )
      printf( "Error received setting sound speed\n" );
    else
      syslog( LOG_ERR, "Error received setting sound speed" );

    close( DspFd );
    return( -1 );
  }
  return( 0 );
}


/*
 * Main entrypoint.
 */
int main( int argc, char *argv[] )
{
  int OptErrors, chr, i, Max;
  char *progname;
  double d;
  char SoundDirectory[ 256 ] = "./sounds";
  short *sPnt, *ssPnt;
  unsigned char *cPnt, *csPnt;

  /*
   * Check the command line arguments.
   */
  progname = argv[ 0 ];
  OptErrors = 0;
  while( (chr = getopt( argc, argv, "d:ps:" )) != EOF) {
    switch( chr ) {

      /*
       * Sound file diretory.
       */
      case 's':
	strncpy( SoundDirectory, optarg, sizeof SoundDirectory );
	if( SoundDirectory[ strlen( SoundDirectory ) - 1 ] == '/' )
	  SoundDirectory[ strlen( SoundDirectory ) - 1 ] = '\0';

        break;

      /*
       * Set debug mode.
       */
      case 'd':
	if( sscanf( optarg, "%d", &Debug ) != 1 ) {
	  printf( "Invalid Debug level\n" );
	  OptErrors++;
	}
	break;

      /*
       * Sync to pps signal.
       */
      case 'p':
	UsePps = 1;
	break;

      default:
        OptErrors++;
        break;
    }
    if( OptErrors ) {
      printf( "usage: %s [ -d # ][ -p ][ -s sound file directory ]\n",
	      progname );
      exit( 1 );
    }
  }
  /*
   * Load all .wav files containing the sounds.
   */
  if( LoadSounds( SoundDirectory ) != 0 ) {
    printf( "Can't load sound files\n" );
    exit( 1 );
  }
  /*
   * Calculate size of synthetic sounds.
   */
  PipLength = (int)(LEN_PIP * (double)(SoundSpeed) * (double)(SoundBytes));
  PauseLength = (int)(LEN_PAUSE * (double)(SoundSpeed) * (double)(SoundBytes));
  /*
   * Generate a single cycle sine wave.
   */
  Max = ( SoundSpeed / 1000 );
  cPnt = (unsigned char *)(SineWave);
  sPnt = (short *)(SineWave);
  for( i = 0,d = 0.0; i < Max; i++,d += M_PI / (double)(Max / 2) ) {
    if( SoundBytes > 1 )
      *sPnt++ = (short)(sin( d ) * 20000.0);
    else
      *cPnt++ = (unsigned char)(sin( d ) * 63.0);
  }
  /*
   * Create a small silent pause sound.
   */
  cPnt = (unsigned char *)(Pause);
  sPnt = (short *)(Pause);
  for( i = 0; i < PauseLength / SoundBytes; i++ ) {
    if( SoundBytes > 1 )
      *sPnt++ = 0;
    else
      *cPnt++ = 127;
  }
  /*
   * Create the beep sound padded with silence.
   */
  cPnt = (unsigned char *)(Pip);
  sPnt = (short *)(Pip);
  for( i = 0; i < PipLength / SoundBytes; i++ ) {
    if( SoundBytes > 1 )
      *sPnt++ = 0;
    else
      *cPnt++ = 127;
  }
  csPnt = (unsigned char *)(SineWave);
  ssPnt = (short *)(SineWave);
  cPnt = (unsigned char *)(Pip);
  sPnt = (short *)(Pip);
  for( i = 0; i < Max * 100; i++ ) {
    if( SoundBytes > 1 )
      *sPnt++ = ssPnt[ i % Max ];
    else
      *cPnt++ = csPnt[ i % Max ] + 128;
  }
  /*
   * If not in debug mode, disconnect the controlling tty
   * and enter daemon mode.
   */
  if( !Debug )
    DisconnectFromTty();

  /*
   * Init syslog.
   */
  openlog( "frokenur", LOG_PID | LOG_NDELAY, LOG_LOCAL0 );

  /*
   * Open PPS device file.
   */
  if( UsePps ) {
    if( ( PpsFd = open( "/dev/ppsd0", O_RDWR ) ) < 0 ) {
      if( Debug )
	printf( "Can't open pps device\n" );
      else
	syslog( LOG_ERR, "Can't open pps device\n" );

      exit( 1 );
    }
  }
  if( OpenSound( "/dev/dsp" ) != 0 ) {
    if( UsePps )
      close( PpsFd );

    exit( 1 );
  }
  syslog( LOG_ERR, "Starting frokenur" );
  /*
   * Play all sounds in debug > 2 mode.
   */
  if( Debug > 2 ) {
    for( i = 0; i < N_SOUNDFILES; i++ ) {
      PlaySound( i );
      write( DspFd, Pause, PauseLength );
    }
    ioctl( DspFd, SNDCTL_DSP_SYNC, 0 );
    sleep( 1 );
  }
  /*
   * Redirect termination signals.
   */
  signal( SIGINT, Terminate );
  signal( SIGPIPE, Terminate );
  /*
   * Enter main loop.
   */
  for( ; ; ) {
    WaitUntil( START_SECOND );
    PlayTime();
    WaitUntilZero();
    PlayBeep();
  }
  return 0;
}
