#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <sys/param.h>

enum {	NO_MESSAGE = 0,
	MAYBE_MESSAGE,
	START_OF_MESSAGE,
	MAYBE_POS_MESSAGE,
	POS_MESSAGE
};

struct termios savedtty;
int ttyfd;
FILE *fp = NULL;

/*
 * Terminate the program. ( parent ).
 */
static void Terminate( int sig )
{

  signal( SIGTERM, SIG_DFL );
  signal( SIGINT, SIG_DFL );
  signal( SIGQUIT, SIG_DFL );
  tcsetattr( ttyfd, TCSANOW, &savedtty );
  close( ttyfd );
  if( fp != NULL )
    fclose( fp );
  printf( "\nTerminating\n" );
  exit( 1 );

}

int InitPort( char *Port, speed_t Speed, struct termios *SavedMode )
{
  int fd;
  struct termios ttymode;

  if( ( fd = open( Port, O_RDWR ) ) < 0 ) {
    printf( "Can't open %s, errno : %d\n", Port, errno );
    return( -1 );
  }
  if( tcgetattr( fd, &ttymode ) < 0 ) {
    printf( "Can't read attributes from %s, errno : %d\n", Port, errno );
    close( fd );
    return( -1 );
  }
  bcopy( &ttymode, SavedMode, sizeof ttymode );
  cfmakeraw( &ttymode );
  cfsetispeed( &ttymode, Speed );
  cfsetospeed( &ttymode, Speed );
  ttymode.c_iflag |= IGNPAR;
  ttymode.c_cflag &= ~PARENB;
  ttymode.c_cflag |= (CS8 | CLOCAL);
  ttymode.c_cflag &= ~(CRTS_IFLOW | CDTR_IFLOW);
  ttymode.c_cc[ VMIN ] = 1;
  ttymode.c_cc[ VTIME ] = 0;
  if( tcsetattr( fd, TCSANOW, &ttymode ) < 0 ) {
    printf( "Can't set attributes on %s, errno : %d\n", Port, errno );
    close( fd );
    return( -1 );
  }
  return( fd );

}

void ToString( int iVal, char *String )
{
  int Deg, Min, Sec, Frac;

  Deg = iVal / 3600000;
  iVal = iVal % 3600000;
  Min = iVal / 60000; 
  iVal = iVal % 60000; 
  Sec = iVal / 1000; 
  Frac = iVal % 1000; 
  sprintf( String, "%3d:%02d:%02d.%03d", Deg, Min, Sec, Frac );

}


void DisplayMessage( unsigned char *Message, int Length )
{
  int Lat, Long, HeightGps, HeightMsl, Veloc, Heading, nSats, tSats;
  int Year, Month, Day, Hour, Min, Sec, Frac;
  char LatString[ 64 ], LongString[ 64 ];

  Year = ( Message[ 2 ] << 8 ) + Message[ 3 ];
  Month = Message[ 0 ];
  Day = Message[ 1 ];
  Hour = Message[ 4 ];
  Min = Message[ 5 ];
  Sec = Message[ 6 ];
  Frac = ( Message[ 7 ] << 24 ) +
	 ( Message[ 8 ] << 16 ) +
	 ( Message[ 9 ] << 8 ) +
	 Message[ 10 ];

  Lat = ( Message[ 13 ] << 24 ) +
	( Message[ 14 ] << 16 ) +
	( Message[ 15 ] << 8 ) +
	Message[ 16 ];
  Long = ( Message[ 17 ] << 24 ) +
	 ( Message[ 18 ] << 16 ) +
	 ( Message[ 19 ] << 8 ) +
	 Message[ 20 ];
  HeightGps = ( Message[ 21 ] << 24 ) +
	      ( Message[ 22 ] << 16 ) +
	      ( Message[ 23 ] << 8 ) +
	      Message[ 24 ];
  HeightMsl = ( Message[ 25 ] << 24 ) +
	      ( Message[ 26 ] << 16 ) +
	      ( Message[ 27 ] << 8 ) +
	      Message[ 28 ];
  Veloc = ( Message[ 29 ] << 8 ) + Message[ 30 ];
  Heading = ( Message[ 31 ] << 8 ) + Message[ 32 ];
  nSats = Message[ 36 ];
  tSats = Message[ 37 ];

  printf( "%04u-%02u-%02u %02u:%02u:%02u.%09u\n",
	  Year, Month, Day, Hour, Min, Sec, Frac );
  ToString( Lat, LatString );
  ToString( Long, LongString );
  Hour = HeightGps / 100;
  Min = abs( HeightGps ) % 100;
  printf( "  Lat %s   Long %s   Height %5d.%d\n",
	  LatString, LongString, Hour, Min );
  printf( "  Visible Sats %d   Tracked Sats %d   Status %02x\n",
	  nSats, tSats, Message[ 70 ] );
}


void ProcessData( unsigned char *Buffer, int Length )
{
  static int gState = NO_MESSAGE;
  static int gLen = 0;
  static unsigned char gBuffer[ 1024 ];

  while( Length-- > 0 ) {
    switch( gState ) {
      case NO_MESSAGE:
	if( *Buffer++ == '@' ) {
	  gState = MAYBE_MESSAGE;
	}
	break;

      case MAYBE_MESSAGE:
	if( *Buffer++ == '@' ) {
	  gState = START_OF_MESSAGE;
	}
	else {
	  gState = NO_MESSAGE;
	}
	break;

      case START_OF_MESSAGE:
	if( *Buffer++ == 'E' ) {
	  gState = MAYBE_POS_MESSAGE;
	}
	else {
	  gState = NO_MESSAGE;
	}

      case MAYBE_POS_MESSAGE:
	if( *Buffer++ == 'a' ) {
	  gState = POS_MESSAGE;
	  gLen = 0;
	}
	else {
	  gState = NO_MESSAGE;
	}

      case POS_MESSAGE:
	gBuffer[ gLen++ ] = *Buffer++;
	if( gLen >= 74 ) {
	  DisplayMessage( gBuffer, gLen );
	  gState = NO_MESSAGE;
	}
	break;
    }
  }

}


void WriteMessage( int fd, unsigned char *Msg, int Length )
{
  unsigned char CheckSum;
  int i;

  write( fd, "@@", 2 );
  CheckSum = 0;
  for( i = 0; i < Length; i++ )
    CheckSum ^= Msg[ i ];

  write( fd, Msg, Length );
  write( fd, &CheckSum, 1 );
  write( fd, "\r\n", 2 );

}


int main( int argc, char *argv[] )
{
  fd_set Fds;
  int cnt;
  unsigned char buf[ 1024 ];
  int Interval = 1;

  if( argc > 1 ) {
    if( sscanf( argv[ 1 ], "%d", &Interval ) != 1
    || Interval < 1 || Interval > 255 ) {
#if 0
      printf( "Invalid interval argument\n" );
      exit( 1 );
#endif
    }
  }
  if( ( ttyfd = InitPort( "/dev/cuaa0", 9600, &savedtty ) ) < 0 ) {
    exit( 1 );
  }
  /*
   * Redirect some signals.
   */
  (void)signal( SIGHUP, SIG_IGN );
  (void)signal( SIGTERM, Terminate );
  (void)signal( SIGINT, Terminate );
  (void)signal( SIGQUIT, Terminate );
#if 0
  WriteMessage( ttyfd, "Cf", 2 );
  usleep( 500000 );
#endif
  buf[ 0 ] = 'E';
  buf[ 1 ] = 'a';
  buf[ 2 ] = (char)(Interval);
  WriteMessage( ttyfd, buf, 3 );
  FD_ZERO( &Fds );
  while( 1 ) {
    FD_SET( ttyfd, &Fds );
    if( select( FD_SETSIZE, &Fds, NULL, NULL, NULL ) < 0 ) {
      printf( "select failed with errno : %d\n", errno );
      exit( 1 );
    }
    if( FD_ISSET( ttyfd, &Fds ) ) {
      cnt = read( ttyfd, buf, sizeof buf );
      ProcessData( buf, cnt );
    }
  }
#if 0
  tcsetattr( ttyfd0, TCSANOW, &savedtty0 );
  close( ttyfd0 );
  tcsetattr( ttyfd1, TCSANOW, &savedtty1 );
  close( ttyfd1 );
  fclose( fp );
  exit( 0 );
#endif
  return 0;
}
