/*
 * microd.h
 * 
 * Header file for the simple microstepper daemon.
 */
#ifndef _MICROD_H_
#define _MICROD_H_


/*
 * Default tcp port used.
 */
#define MICROD_PORT 273


/*
 * Default time between microstepper updates. ( in seconds ).
 */
#define MICROD_UPDATE_TIME 15


/*
 * Command codes received from clients.
 */
enum {	CLK_ECODE = 0,
	CLK_OFFSET,
	CLK_GETTIMESTAMP,
	CLK_TIMESTAMP,
	CLK_GETMICROSTEP,
	CLK_MICROSTEP,
	CLK_SETMICROSTEP,
	CLK_GETPPSOUTPUT,
	CLK_PPSOUTPUT,
	CLK_SETPPSOUTPUT,
	CLK_GETSTAT,
	CLK_STAT,
	CLK_GETMED,
	CLK_MED,
	CLK_GETHZ,
	CLK_HZ,
	CLK_GETCLKMODE,
	CLK_CLKMODE,
	CLK_DRIFT
};


/*
 * Error codes sent to clients.
 */
enum {	OK = 0,
	INVALID_CMD,
	GARBAGE_CMD,
	IOCTLERROR,
	NO_PPS,
	INVALID_OFFSET,
	INVALID_DRIFT,
	INVALID_LEAP,
	INVALID_STATUS
};


/*
 * Internal error codes.
 */
enum {	MEMORY_ERROR = 1,
	UNKNOWN_KEY_TYPE,
	NO_KEY_FILE,
	NO_CONFIG_FILE,
	CONFIG_ERROR,
	INVALID_INTERVAL,
	INVALID_PORT,
	INVALID_MAXOFFSET,
	INVALID_MAXDISPERSION,
	INVALID_PPSDEVICE,
	INVALID_OFFSETFILE,
	NO_INTERVAL,
	NO_PORT,
	NO_DEVADDR,
	NO_MAXOFFSET,
	NO_MAXDISPERSION,
	NO_PPSDEVICE,
	NO_OFFSETFILE,
	NO_USESSL,
	NO_CERTFILE,
	NO_KEYFILE,
	NO_CAFILE,
	NO_CIPHER,
	NO_NTPFILTER,
	NO_ALLOWRESP,
	NO_ALLOWCONTROL,
	MAX_ERROR_NUMBER
};


/*
 * Types of ascii representations for keys.  "Standard" means a 64 bit
 * hex number in NBS format, i.e. with the low order bit of each byte
 * a parity bit.  "NTP" means a 64 bit key in NTP format, with the
 * high order bit of each byte a parity bit.  "Ascii" means a 1-to-8
 * character string whose ascii representation is used as the key.
 */

#define	KEY_TYPE_STD	1
#define	KEY_TYPE_NTP	2
#define	KEY_TYPE_ASCII	3
#define	KEY_TYPE_MD5	4

/*
 * Maximum key size.
 */
#define	MAX_KEY_LENGTH 8

/*
 * Structure to hold a MD5 key.
 */
struct NtpKey {
	struct NtpKey	*next;
	u_long		KeyId;		/* Key id in ntp packets. */
	int		KeyType;	/* Key type defined below. */
	int		KeyLength;	/* Length of key. */
	u_char		Key[ MAX_KEY_LENGTH + 1 ];
};

/*
 * Ntp packet types.
 */
#define NTP_QUERY_PKT    3
#define NTP_RESPONSE_PKT 4
#define NTP_CONTROL_PKT  7

/*
 * Microstepper offset data block.
 */
struct ClockData { int		Offset;		/* Signed offset in ns    */
		   int		Drift;		/* Signed drift in ps/24h */
		   short	Leap;		/* Leap second. -1, 0, +1 */
		   short	Status;		/* Other status bits      */
};

/*
 * Microstepper offset data block with offset/drift/freq drift.
 */
struct ClockDriftData { int	Offset;		/* Signed offset in ns    */
			int	Drift;		/* Signed drift in ps/24h */
			int	FreqDrift;	/* Signed fdrift in fs/s/24h */
			short	Leap;		/* Leap second. -1, 0, +1 */
			short	Status;		/* Other status bits      */
};

/*
 * Union of all commands.
 */
union Cmds { short			Ecode;		/* Error code        */
	     int			NanoSec;	/* Time in ns	     */
	     int			MicroSec;	/* Time in us	     */
	     int			Hz;		/* Frequency	     */
	     int			Mode;		/* Clock mode	     */
	     struct ppsstat		Stat;		/* Pps statistics    */
	     struct ppsclockev		ClockEvent;	/* Ev from pps driver*/
	     struct ClockData		ClkStatus;	/* Offset data block */
	     struct ClockDriftData	ClkDrift;	/* Drift data block  */
};

/*
 * Complete command structure.
 */
struct ClkCmd { short		Length;		/* Size of Data + Type    */
		short		Type;		/* Command type code      */
		union Cmds	Data;		/* Command data block     */
};

/*
 * Receive states.
 */
enum {	RCV_LENGTH = 0,
	RCV_DATA
};

/*
 * Data structure holding state for a connected client.
 */
struct ClkClient { struct ClkClient	*Next;		/* Link field	     */
		   int			Sock;		/* Rcv socket	     */
		   int			State;		/* Rcv state	     */
		   SSL			*Ssl;		/* Ssl state	     */
		   int			ExpectedLength;	/* Data block length */
		   int			BufLength;	/* Amount received   */
		   unsigned char	Buffer[ 1024 ];	/* Rcv data storage  */
};

/*
 * Microd configuration structure.
 */
struct Config { int		StepperInterval;
		int		UseSsl;
		int		NtpFilter;
		int		AllowResponses;
		int		AllowControl;
		unsigned short	Port;
		unsigned long	DeviceAddress;
		double		MaxOffset;
		double		MaxDispersion;
		char		PpsDevice[ 64 ];
		char		OffsetFile[ 128 ];
		char		SslCa[ 128 ];
		char		SslCert[ 128 ];
		char		SslKey[ 128 ];
		char		SslCipher[ 64 ];
};

/*
 * Prototypes.
 */
int ReadConfig( char *ConfigFile, int *LineNo );

/*
 * Global data used by the microd daemon.
 */

#ifdef DEFINE_GLOBALS
struct NtpKey *NtpKeyChain = NULL;
int nNtpKeys = 0;
struct Config MicrodConfig;
char *ErrorTexts[] =
{
  "No error",						/*   0 */
  "Memory allocation error",				/*   1 */
  "Unknown ntp key type",				/*   2 */
  "No ntp key file found",				/*   3 */
  "No configuration file found",			/*   4 */
  "Syntax error in configuration file",			/*   5 */
  "Invalid stepper interval",				/*   6 */
  "Invalid port",					/*   7 */
  "Invalid maxoffset",					/*   8 */
  "Invalid maxdispersion",				/*   9 */
  "Invalid ppsdevice",					/*  10 */
  "Invalid offsetfile",					/*  11 */
  "No stepper interval specified",			/*  12 */
  "No port specified",					/*  13 */
  "No device address specified",			/*  14 */
  "No maxoffset specified",				/*  15 */
  "No maxdispersion specified",				/*  16 */
  "No ppsdevice specified",				/*  17 */
  "No offsetfile specified",				/*  18 */
  "Ssl usage not specified",				/*  19 */
  "No certificate file specified",			/*  20 */
  "No private key file specified",			/*  21 */
  "No CA file specified",				/*  22 */
  "No cipher specified",				/*  23 */
  "Ntp filter usage not specified",			/*  24 */
  "Ntp response msg handling not specified",		/*  25 */
  "Ntp control msg handling not specified",		/*  26 */
  "Unspecified error"					/* Max */
};
#else
extern struct NtpKey *NtpKeyChain;
extern int nNtpKeys;
extern struct Config MicrodConfig;
extern char *ErrorTexts[];
#endif

#endif /* _MICROD_H_ */
