/*
 * File : microapi.c
 *
 * Date : 000128
 *
 * API to interface a application to microd daemons.
 *
 * The communication can be performed using cleartext or ssl.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <openssl/rsa.h>
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/e_os.h>
#include <openssl/buffer.h>
#include <openssl/bio.h>

#include "ppsclock.h"
#include "microd.h"
#include "microapi.h"

static char SslCertFile[ 256 ] = "/etc/certificate/Microd_cert.pem";
static char SslKeyFile[ 256 ] = "/etc/certificate/Microd_key.pem";
static char SslCaFile[ 256 ] = "/etc/certificate/Cacert.pem";
static char SslCipher[ 256 ] = "RC4-MD5";

static SSL_CTX *SslCtx = NULL;
static SSL_METHOD *SslMethod = NULL;


int InitSsl( char *CertFile, char *KeyFile, char *CaFile, char *Cipher )
{

  if( CertFile != NULL )
    strncpy( SslCertFile, CertFile, sizeof SslCertFile );

  if( KeyFile != NULL )
    strncpy( SslKeyFile, KeyFile, sizeof SslKeyFile );

  if( CaFile != NULL )
    strncpy( SslCaFile, CaFile, sizeof SslCaFile );

  if( Cipher != NULL )
    strncpy( SslCipher, Cipher, sizeof SslCipher );

  SSL_load_error_strings();
  SSLeay_add_ssl_algorithms();
  SslMethod = SSLv23_client_method();
  SslCtx = SSL_CTX_new( SslMethod );
  if( SslCtx == NULL )
    return( MICROD_SSLCTX_ERROR );

  SSL_CTX_set_options( SslCtx, 0 );
  if( SSL_CTX_use_certificate_file( SslCtx, SslCertFile,
				    SSL_FILETYPE_PEM ) <= 0 ) {
    SSL_CTX_free( SslCtx );
    SslCtx = NULL;
    return( MICROD_SSLCERT_ERROR );
  }
  if( SSL_CTX_use_PrivateKey_file( SslCtx, SslKeyFile,
				   SSL_FILETYPE_PEM ) <= 0 ) {
    SSL_CTX_free( SslCtx );
    SslCtx = NULL;
    return( MICROD_SSLKEY_ERROR );
  }
  SSL_CTX_set_cipher_list( SslCtx, SslCipher );
  if( ( !SSL_CTX_load_verify_locations( SslCtx, SslCaFile, NULL ) )
  || ( !SSL_CTX_set_default_verify_paths( SslCtx ) ) ) {
    SSL_CTX_free( SslCtx );
    SslCtx = NULL;
    return( MICROD_SSLVERIFY_ERROR );
  }
  return( 0 );

}


/*
 * Use gethostbyname to find the ip addr of the host.
 */
int FindIpAddr( char *HostName, unsigned long *IpAddr )
{
  struct hostent *Host;

  if( ( Host = gethostbyname( HostName ) ) == NULL )
    return( MICROD_HOSTNAME_ERROR );

  *IpAddr = *((unsigned long *)(Host->h_addr));
  return( 0 );

}


/*
 * Cleartext versions.
 */

/*
 * Create and open a socket to the host, send offset and
 * drift values and then disconnect.
 * Leap and status fields are zeroed.
 */
int SendDriftAndOffset( unsigned long IpAddr, int Offset, int Drift )
{
  int Sock, Len;
  struct ClkCmd *CmdPnt;
  struct sockaddr_in Sa;
  char Buf[ 1024 ];


  /*
   * Create a stream socket.
   */
  Sock = socket(AF_INET, SOCK_STREAM, 0);
  if( Sock < 0 ) {
    return( MICROD_SOCKET_ERROR );
  }

  /*
   * Connect to the ntp server.
   */
  bzero( &Sa, sizeof Sa );
  Sa.sin_family = AF_INET;
  Sa.sin_port =  htons( MICROD_PORT );
  Sa.sin_addr.s_addr = IpAddr;
  if( connect( Sock, (struct sockaddr *)(&Sa), sizeof(Sa) ) < 0 ) {
    close( Sock );
    return( MICROD_CONNECT_ERROR );
  }

  /*
   * Format a command in the buffer.
   */
  CmdPnt = (struct ClkCmd *)(Buf);
  Len = sizeof( short ) + sizeof( struct ClockData );
  CmdPnt->Length = htons( Len );
  CmdPnt->Type = htons( CLK_OFFSET );
  CmdPnt->Data.ClkStatus.Offset = htonl( Offset );
  CmdPnt->Data.ClkStatus.Drift = htonl( Drift );
  CmdPnt->Data.ClkStatus.Leap = htons( 0 );
  CmdPnt->Data.ClkStatus.Status = htons( 0 );

  /*
   * Send the command to the ntp server.
   */
  write( Sock, Buf, Len + sizeof( short ) );

  /*
   * Wait for the return status code.
   */
  if( read( Sock, Buf, 3 * sizeof( short ) ) != 3 * sizeof( short ) ) {
    close( Sock );
    return( MICROD_READ_ERROR );
  }

  /*
   * Examine the received data.
   */
  if( ntohs( CmdPnt->Type ) != CLK_ECODE ) {
    close( Sock );
    return( MICROD_TYPE_ERROR );
  }
  if( ntohs( CmdPnt->Data.Ecode ) != OK ) {
    close( Sock );
    return( (int)(ntohs( CmdPnt->Data.Ecode )) );
  }
  close( Sock );
  return( 0 );

}


/*
 * Create and open a socket to the host, send offset, drift
 * and frquency drift values and then disconnect.
 * Leap and status fields are zeroed.
 */
int SendFreqDriftAndOffset( unsigned long IpAddr, int Offset,
			    int Drift, int FreqDrift )
{
  int Sock, Len;
  struct ClkCmd *CmdPnt;
  struct sockaddr_in Sa;
  char Buf[ 1024 ];


  /*
   * Create a stream socket.
   */
  Sock = socket(AF_INET, SOCK_STREAM, 0);
  if( Sock < 0 ) {
    return( MICROD_SOCKET_ERROR );
  }

  /*
   * Connect to the ntp server.
   */
  bzero( &Sa, sizeof Sa );
  Sa.sin_family = AF_INET;
  Sa.sin_port =  htons( MICROD_PORT );
  Sa.sin_addr.s_addr = IpAddr;
  if( connect( Sock, (struct sockaddr *)(&Sa), sizeof(Sa) ) < 0 ) {
    close( Sock );
    return( MICROD_CONNECT_ERROR );
  }

  /*
   * Format a command in the buffer.
   */
  CmdPnt = (struct ClkCmd *)(Buf);
  Len = sizeof( short ) + sizeof( struct ClockDriftData );
  CmdPnt->Length = htons( Len );
  CmdPnt->Type = htons( CLK_DRIFT );
  CmdPnt->Data.ClkDrift.Offset = htonl( Offset );
  CmdPnt->Data.ClkDrift.Drift = htonl( Drift );
  CmdPnt->Data.ClkDrift.FreqDrift = htonl( FreqDrift );
  CmdPnt->Data.ClkDrift.Leap = htons( 0 );
  CmdPnt->Data.ClkDrift.Status = htons( 0 );

  /*
   * Send the command to the ntp server.
   */
  write( Sock, Buf, Len + sizeof( short ) );

  /*
   * Wait for the return status code.
   */
  if( read( Sock, Buf, 3 * sizeof( short ) ) != 3 * sizeof( short ) ) {
    close( Sock );
    return( MICROD_READ_ERROR );
  }

  /*
   * Examine the received data.
   */
  if( ntohs( CmdPnt->Type ) != CLK_ECODE ) {
    close( Sock );
    return( MICROD_TYPE_ERROR );
  }
  if( ntohs( CmdPnt->Data.Ecode ) != OK ) {
    close( Sock );
    return( (int)(ntohs( CmdPnt->Data.Ecode )) );
  }
  close( Sock );
  return( 0 );

}


/*
 * SSL version.
 *
 * Create and open a socket to the host, send offset and
 * drift values and then disconnect.
 * Leap and status fields are zeroed.
 */
int SslSendDriftAndOffset( unsigned long IpAddr, int Offset, int Drift )
{
  int Sock, Len;
  struct ClkCmd *CmdPnt;
  struct sockaddr_in Sa;
  SSL *Ssl;
  char Buf[ 1024 ];


  if( SslCtx == NULL )
    return( MICROD_SSLNOTINIT_ERROR );

  /*
   * Create a stream socket.
   */
  Sock = socket(AF_INET, SOCK_STREAM, 0);
  if( Sock < 0 ) {
    return( MICROD_SOCKET_ERROR );
  }

  /*
   * Connect to the ntp server.
   */
  bzero( &Sa, sizeof Sa );
  Sa.sin_family = AF_INET;
  Sa.sin_port =  htons( MICROD_PORT );
  Sa.sin_addr.s_addr = IpAddr;
  if( connect( Sock, (struct sockaddr *)(&Sa), sizeof(Sa) ) < 0 ) {
    close( Sock );
    return( MICROD_CONNECT_ERROR );
  }
  Ssl = SSL_new( SslCtx );
  if( Ssl == NULL )
    return( MICROD_SSLALLOC_ERROR );
  SSL_set_fd( Ssl, Sock );
  SSL_connect( Ssl );

  /*
   * Format a command in the buffer.
   */
  CmdPnt = (struct ClkCmd *)(Buf);
  Len = sizeof( short ) + sizeof( struct ClockData );
  CmdPnt->Length = htons( Len );
  CmdPnt->Type = htons( CLK_OFFSET );
  CmdPnt->Data.ClkStatus.Offset = htonl( Offset );
  CmdPnt->Data.ClkStatus.Drift = htonl( Drift );
  CmdPnt->Data.ClkStatus.Leap = htons( 0 );
  CmdPnt->Data.ClkStatus.Status = htons( 0 );

  /*
   * Send the command to the ntp server.
   */
  SSL_write( Ssl, Buf, Len + sizeof( short ) );

  /*
   * Wait for the return status code.
   */
  if( SSL_read( Ssl, Buf, 3 * sizeof( short ) ) != 3 * sizeof( short ) ) {
    SSL_shutdown( Ssl );
    shutdown( SSL_get_fd( Ssl ), 2 );
    close( Sock );
    if( Ssl != NULL )
      SSL_free( Ssl );

    return( MICROD_READ_ERROR );
  }

  /*
   * Examine the received data.
   */
  if( ntohs( CmdPnt->Type ) != CLK_ECODE ) {
    SSL_shutdown( Ssl );
    shutdown( SSL_get_fd( Ssl ), 2 );
    close( Sock );
    if( Ssl != NULL )
      SSL_free( Ssl );

    return( MICROD_TYPE_ERROR );
  }
  if( ntohs( CmdPnt->Data.Ecode ) != OK ) {
    SSL_shutdown( Ssl );
    shutdown( SSL_get_fd( Ssl ), 2 );
    close( Sock );
    if( Ssl != NULL )
      SSL_free( Ssl );

    return( (int)(ntohs( CmdPnt->Data.Ecode )) );
  }
  SSL_shutdown( Ssl );
  shutdown( SSL_get_fd( Ssl ), 2 );
  close( Sock );
  if( Ssl != NULL )
    SSL_free( Ssl );

  return( 0 );

}

/*
 * Create and open a socket to the host, send offset, drift
 * and frequency drift values and then disconnect.
 * Leap and status fields are zeroed.
 */
int SslSendFreqDriftAndOffset( unsigned long IpAddr, int Offset,
			       int Drift, int FreqDrift )
{
  int Sock, Len;
  struct ClkCmd *CmdPnt;
  struct sockaddr_in Sa;
  SSL *Ssl;
  char Buf[ 1024 ];


  if( SslCtx == NULL )
    return( MICROD_SSLNOTINIT_ERROR );

  /*
   * Create a stream socket.
   */
  Sock = socket(AF_INET, SOCK_STREAM, 0);
  if( Sock < 0 ) {
    return( MICROD_SOCKET_ERROR );
  }

  /*
   * Connect to the ntp server.
   */
  bzero( &Sa, sizeof Sa );
  Sa.sin_family = AF_INET;
  Sa.sin_port =  htons( MICROD_PORT );
  Sa.sin_addr.s_addr = IpAddr;
  if( connect( Sock, (struct sockaddr *)(&Sa), sizeof(Sa) ) < 0 ) {
    close( Sock );
    return( MICROD_CONNECT_ERROR );
  }
  Ssl = SSL_new( SslCtx );
  if( Ssl == NULL )
    return( MICROD_SSLALLOC_ERROR );
  SSL_set_fd( Ssl, Sock );
  SSL_connect( Ssl );

  /*
   * Format a command in the buffer.
   */
  CmdPnt = (struct ClkCmd *)(Buf);
  Len = sizeof( short ) + sizeof( struct ClockDriftData );
  CmdPnt->Length = htons( Len );
  CmdPnt->Type = htons( CLK_DRIFT );
  CmdPnt->Data.ClkDrift.Offset = htonl( Offset );
  CmdPnt->Data.ClkDrift.Drift = htonl( Drift );
  CmdPnt->Data.ClkDrift.FreqDrift = htonl( FreqDrift );
  CmdPnt->Data.ClkDrift.Leap = htons( 0 );
  CmdPnt->Data.ClkDrift.Status = htons( 0 );

  /*
   * Send the command to the ntp server.
   */
  SSL_write( Ssl, Buf, Len + sizeof( short ) );

  /*
   * Wait for the return status code.
   */
  if( SSL_read( Ssl, Buf, 3 * sizeof( short ) ) != 3 * sizeof( short ) ) {
    SSL_shutdown( Ssl );
    shutdown( SSL_get_fd( Ssl ), 2 );
    close( Sock );
    if( Ssl != NULL )
      SSL_free( Ssl );

    return( MICROD_READ_ERROR );
  }

  /*
   * Examine the received data.
   */
  if( ntohs( CmdPnt->Type ) != CLK_ECODE ) {
    SSL_shutdown( Ssl );
    shutdown( SSL_get_fd( Ssl ), 2 );
    close( Sock );
    if( Ssl != NULL )
      SSL_free( Ssl );

    return( MICROD_TYPE_ERROR );
  }
  if( ntohs( CmdPnt->Data.Ecode ) != OK ) {
    SSL_shutdown( Ssl );
    shutdown( SSL_get_fd( Ssl ), 2 );
    close( Sock );
    if( Ssl != NULL )
      SSL_free( Ssl );

    return( (int)(ntohs( CmdPnt->Data.Ecode )) );
  }
  SSL_shutdown( Ssl );
  shutdown( SSL_get_fd( Ssl ), 2 );
  close( Sock );
  if( Ssl != NULL )
    SSL_free( Ssl );

  return( 0 );

}
