/*
 * Simple microstepper daemon.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <syslog.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <signal.h>
#include <errno.h>
#include <math.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ntp_fp.h>
#include <ntp.h>
#include <ntp_request.h>
#include <ntp_stdlib.h>
#include <lib_strbuf.h>
#include <ntp_string.h>
#include <ntp_unixtime.h>

#include <openssl/rsa.h>
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/e_os.h>
#include <openssl/buffer.h>
#include <openssl/bio.h>

#include "ppsclock.h"
#include "checkntp.h"

#define DEFINE_GLOBALS
#include "microd.h"

int PpsFd = -1;
int IoctlSock = -1;
int NtpSock = -1;
int ClkOffset = 0;
int ClkDrift = 0;
int ClkFreqDrift = 0;
int Syncronized = 0;
long SyncTimer = 0L;
long LastStepperUpdate = 0L;
time_t LastReceivedOffset = 0L;
struct ClkClient *Clients = NULL;
fd_set AllFds;
char ConfigFile[ 128 ] = "/etc/microd.conf";

SSL_CTX* SslCtx;
SSL_METHOD *SslMeth;
int SslSessionIdContext = 1; /* anything will do */
unsigned char GlobalBuf[ 16384 ];


/*
 * Terminate the program.
 */
static void Terminate( int sig )
{

  signal( SIGTERM, SIG_DFL );
  syslog( LOG_ERR, "Terminating" );
  if( PpsFd >= 0 )
    close( PpsFd );

  exit( 1 );

}

/*
 * Ssl callback routines.
 */
static int verify_callback( int ok, X509_STORE_CTX *ctx )
{
  char buf[256];
  X509 *err_cert;
  int err,depth;

  err_cert = X509_STORE_CTX_get_current_cert( ctx );
  err = X509_STORE_CTX_get_error( ctx );
  depth = X509_STORE_CTX_get_error_depth( ctx );

  X509_NAME_oneline( X509_get_subject_name(err_cert), buf, sizeof buf );
#if 0
  //  syslog(LOG_NOTICE,"depth=%d %s",depth,buf);
  //  BIO_printf(bio_err,"depth=%d %s\n",depth,buf);
#endif
  if( !ok ) {
    syslog( LOG_WARNING, "verify error:num=%d:%s",err,
            X509_verify_cert_error_string( err ) );
    }
#if 0
  switch (ctx->error)
    {
    case X509_V_ERR_UNABLE_TO_GET_ISSUER_CERT:
      X509_NAME_oneline(X509_get_issuer_name(ctx->current_cert),buf,256);
      BIO_printf(bio_err,"issuer= %s\n",buf);
      break;
    case X509_V_ERR_CERT_NOT_YET_VALID:
    case X509_V_ERR_ERROR_IN_CERT_NOT_BEFORE_FIELD:
      BIO_printf(bio_err,"notBefore=");
      ASN1_TIME_print(bio_err,X509_get_notBefore(ctx->current_cert));
      BIO_printf(bio_err,"\n");
      break;
    case X509_V_ERR_CERT_HAS_EXPIRED:
    case X509_V_ERR_ERROR_IN_CERT_NOT_AFTER_FIELD:
      BIO_printf(bio_err,"notAfter=");
      ASN1_TIME_print(bio_err,X509_get_notAfter(ctx->current_cert));
      BIO_printf(bio_err,"\n");
      break;
    }
#endif
  return( ok );

}

static RSA *tmp_rsa_cb( SSL *s, int is_export, int keylength )
{
  static RSA *rsa_tmp=NULL;

  if( rsa_tmp == NULL )
    rsa_tmp = RSA_generate_key( keylength, RSA_F4, NULL, NULL );

  return( rsa_tmp );

}

/*
 * Update the microstepper.
 */
static void SetStepper( int Offset )
{
  time_t Now;

  Now = time( NULL );
  LastStepperUpdate = Now;
  if (ioctl(PpsFd, CIOSETSTEP, (caddr_t)&Offset) < 0) {
    syslog( LOG_ERR, "Can't set MicroStep (%m)" );
    return;
  }
#if 0
  syslog( LOG_ERR, "Offset: %d", Offset );
#endif
}

/*
 * Disable all ntp packet reception.
 */
static int DisallowAllNtp()
{
  int i = 0;

#if 0
  return( ioctl( IoctlSock, SIOCIP_NTPCLEARMODES, (caddr_t)(&i) ) );
#else
  int err;
  err = ioctl( IoctlSock, SIOCIP_NTPCLEARMODES, (caddr_t)(&i) );
  if( err != 0 )
    syslog( LOG_ERR, "ioctl( SIOCIP_NTPCLEARMODES ) returned %d", err );
  return err;
#endif
}

/*
 * Enable reception of a ntp packet type.
 */
static int AllowNtp( int Type )
{

#if 0
  return( ioctl( IoctlSock, SIOCIP_NTPALLOWMODE, (caddr_t)(&Type) ) );
#else
  int err;
  err = ioctl( IoctlSock, SIOCIP_NTPALLOWMODE, (caddr_t)(&Type) );
  if( err != 0 )
    syslog( LOG_ERR, "ioctl( SIOCIP_NTPALLOWMODE ) returned %d", err );
  return err;
#endif

}

/*
 * Disable reception of a ntp packet type.
 */
static int DisallowNtp( int Type )
{

#if 0
  return( ioctl( IoctlSock, SIOCIP_NTPDISALLOWMODE, (caddr_t)(&Type) ) );
#else
  int err;
  err = ioctl( IoctlSock, SIOCIP_NTPDISALLOWMODE, (caddr_t)(&Type) );
  if( err != 0 )
    syslog( LOG_ERR, "ioctl( SIOCIP_NTPDISALLOWMODE ) returned %d", err );
  return err;
#endif

}

/*
 * Get offset and dispersion from ntpd.
 */
static int GetOffsetAndDispersion( double *Offset, double *Dispersion )
{
  char InputBuffer[ 1024 ];
  char sBuf[ 128 ];
  struct req_pkt *NtpPkt;
  struct info_peer_summary *NtpSum;
  struct sockaddr_in Sa;
  fd_set Fds;
  int i, Length;
  l_fp tempts;
  struct timeval tm;

  bzero( &Sa, sizeof Sa );
  Sa.sin_len = sizeof( struct sockaddr_in );
  Sa.sin_family = AF_INET;
  Sa.sin_addr.s_addr = inet_addr( "127.0.0.1" );
  Sa.sin_port =  htons( 123 );
  bzero( InputBuffer, sizeof( struct req_pkt ) );
  NtpPkt = (struct req_pkt *)(InputBuffer);
  NtpPkt->rm_vn_mode = RM_VN_MODE( 0, 0 );
  NtpPkt->auth_seq = AUTH_SEQ( 0, 0 );
  NtpPkt->implementation = IMPL_XNTPD;
  NtpPkt->request = REQ_PEER_LIST_SUM;
  NtpPkt->err_nitems = ERR_NITEMS( 0, 0 );
  NtpPkt->mbz_itemsize = MBZ_ITEMSIZE( 0 );
  if( sendto( NtpSock, InputBuffer, REQ_LEN_NOMAC, 0,
	      (struct sockaddr *)(&Sa), sizeof Sa ) != REQ_LEN_NOMAC ) {
    syslog( LOG_ERR,"Can't send ntp request (%m)" );
    return( 1 );
  }
  tm.tv_sec = 5;
  tm.tv_usec = 0;
  FD_ZERO( &Fds );
  FD_SET( NtpSock, &Fds );
  if( ( i = select( FD_SETSIZE, &Fds, NULL, NULL, &tm ) ) < 0 ) {
    syslog( LOG_ERR, "select failed (%m)" );
    return( 1 );
  }
  /*
   * Timeout.
   */
  if( i == 0 ) {
    syslog( LOG_ERR, "No answer from ntpd" );
    return( 1 );
  }
  if( FD_ISSET( NtpSock, &Fds ) ) {
    /*
     * Answer received.
     */
    i = sizeof Sa;
    bzero( InputBuffer, sizeof InputBuffer );
    Length = recvfrom( NtpSock, InputBuffer, sizeof InputBuffer, 0,
		       (struct sockaddr *)(&Sa), &i );
    if( Length < 0 ) {
      syslog( LOG_ERR, "recvfrom failed (%m)" );
      return( 1 );
    }
    NtpSum = (struct info_peer_summary *)(NtpPkt->data);
    if( INFO_NITEMS( NtpPkt->err_nitems ) <= 0 ) {
      syslog( LOG_ERR, "Received 0 items" );
      return( 1 );
    }
    for( i = 0; i < INFO_NITEMS( NtpPkt->err_nitems); i++ ) {
      if( NtpSum->srcadr == MicrodConfig.DeviceAddress ) {
        if( NtpSum->reach == 0xff ) {
	  NTOHL_FP( &NtpSum->offset, &tempts );
	  sprintf( sBuf, "%-9.9s", lfptoa( &tempts, 6 ) );
	  if( sscanf( sBuf, "%lf", Offset ) != 1 ) {
	    syslog( LOG_ERR, "Invalid offset value received" );
	    return( 1 );
	  }
	  strncpy( sBuf, ufptoa(NTOHS_FP(NtpSum->dispersion), 5),
		   sizeof sBuf );
	  if( sscanf( sBuf, "%lf", Dispersion ) != 1 ) {
	    syslog( LOG_ERR, "Invalid dispersion value received" );
	    return( 1 );
	  }
	}
	else {
	  *Offset = 9.999;
	  *Dispersion = 9.999;
	}
	return( 0 );
      }
      NtpSum++;
    }
    syslog( LOG_ERR, "Can't find ntp device in summary record" );
    return( 1 );
  }
  else {
    syslog( LOG_ERR, "select returns without data" );
  }
  return( 1 );

}

/*
 * Wrapped socket write.
 * Select write or SSL_write according to MicrodConfig.UseSsl.
 */
static int MicrodWrite( struct ClkClient *Client, void *Buf, int Length )
{
  int RetCode;

  if( MicrodConfig.UseSsl ) {
try_again:
    RetCode = SSL_write( Client->Ssl, Buf, Length );
    if( RetCode < 0 ) {
      RetCode = SSL_get_error( Client->Ssl, RetCode );
      if( RetCode == SSL_ERROR_WANT_WRITE ) {
        sleep( 1 );
        goto try_again;
      }
    }
  }
  else {
    RetCode = write( Client->Sock, Buf, Length );
  }
  return( RetCode );
}

/*
 * Wrapped socket read.
 * Select read or SSL_read according to MicrodConfig.UseSsl.
 */
static int MicrodRead( struct ClkClient *Client, void *Buf, int Length )
{
  int RetCode;

  if( MicrodConfig.UseSsl ) {
    for( ; ; ) {
      RetCode = SSL_read( Client->Ssl, Buf, Length );
      switch( SSL_get_error( Client->Ssl, RetCode ) ) {
	case SSL_ERROR_NONE:
	  return( RetCode );

	case SSL_ERROR_WANT_WRITE:
	case SSL_ERROR_WANT_READ:
	case SSL_ERROR_WANT_X509_LOOKUP:
	  sleep( 1 );
	  break;
	case SSL_ERROR_SYSCALL:
	case SSL_ERROR_SSL:
	  return( -1 );
	case SSL_ERROR_ZERO_RETURN:
	  return( 0 );

	default:
	  sleep( 1 );
      }
    }
  }
  else {
    RetCode = read( Client->Sock, Buf, Length );
  }
  return( RetCode );
}

/*
 * Remove all traces of a disconnected client.
 */
static void RemoveClient( struct ClkClient *Client )
{
  struct ClkClient *Clnt;
  struct sockaddr_in Sa;
  int SaLen = sizeof Sa;

  /*
   * Log the connection to syslog.
   */
  if( getpeername( Client->Sock, (struct sockaddr *)(&Sa), &SaLen ) != 0 ) {
    syslog( LOG_ERR, "Error from getpeername() at disconnect, %m" );
  }
  else  {
    if( MicrodConfig.UseSsl )
      syslog( LOG_ERR, "Ssl client disconnected ( %s )",
	      inet_ntoa( Sa.sin_addr ) );
    else
      syslog( LOG_ERR, "Cleartext client disconnected ( %s )",
	      inet_ntoa( Sa.sin_addr ) );
  }
  FD_CLR( Client->Sock, &AllFds );
  if( MicrodConfig.UseSsl && Client->Ssl != NULL ) {
    SSL_set_shutdown( Client->Ssl, SSL_SENT_SHUTDOWN | SSL_RECEIVED_SHUTDOWN );
    SSL_shutdown( Client->Ssl );
    if( Client->Ssl != NULL )
      SSL_free( Client->Ssl );
  }
  close( Client->Sock );
  if( Client == Clients ) {
    Clients = Client->Next;
    free( Client );
    return;
  }
  else {
    for( Clnt = Clients; Clnt->Next != NULL; Clnt = Clnt->Next ) {
      if( Clnt->Next == Client ) {
	Clnt->Next = Client->Next;
	free( Client );
	return;
      }
    }
  }
  syslog( LOG_ERR, "Confusion, can't find client struct in client chain" );

}


/*
 * Insert a new client in the chain of client structs.
 */
static void NewClient( int Sock )
{
  struct ClkClient *Clnt;
  struct sockaddr_in Sa;
  int SaLen = sizeof Sa;

  /*
   * Log the connection to syslog.
   */
  if( getpeername( Sock, (struct sockaddr *)(&Sa), &SaLen ) != 0 ) {
    syslog( LOG_ERR, "Error from getpeername() at connect, %m" );
  }
  else  {
    if( MicrodConfig.UseSsl )
      syslog( LOG_ERR, "New ssl client connected ( %s )",
	      inet_ntoa( Sa.sin_addr ) );
    else
      syslog( LOG_ERR, "New cleartext client connected ( %s )",
	      inet_ntoa( Sa.sin_addr ) );
  }
  if( ( Clnt = (struct ClkClient *)
	       (malloc( sizeof( struct ClkClient ) )) ) == NULL ) {
    syslog( LOG_ERR, "Memory allocation error" );
    close( Sock );
    return;
  }
  bzero( Clnt, sizeof( struct ClkClient ) );
  Clnt->Sock = Sock;
  Clnt->State = RCV_LENGTH;
  Clnt->BufLength = 0;
  Clnt->Next = Clients;
  Clients = Clnt;
  FD_SET( Sock, &AllFds );
  if( MicrodConfig.UseSsl ) {
    Clnt->Ssl = SSL_new( SslCtx );
    SSL_set_fd( Clnt->Ssl, Sock );
    if( SSL_accept( Clnt->Ssl ) <= 0 ) {
      syslog( LOG_ERR, "SSL_accept returns error" );
      RemoveClient( Clnt );
    }
  }
}

/*
 * Send a response code to the connected client.
 */
static void SendResponse( struct ClkClient *Client, short Ecode )
{
  struct ClkCmd Response;

  Response.Length = htons( 2 * sizeof( short ) );
  Response.Type = htons( CLK_ECODE );
  Response.Data.Ecode = htons( Ecode );
  MicrodWrite( Client, &Response, 3 * sizeof( short ) );
  if( Ecode != OK )
    syslog( LOG_ERR, "Error code %d sent", Ecode );

}

/*
 * Calculate a new integer offset.
 */
static int CalcOffset( int Offset, int Drift, int FreqDrift,
		       time_t LastReceivedOffset, time_t Now )
{
  double dDrift;
  double dTimePassed;
  double dTotalOffset;

  dTimePassed = (double)(Now - LastReceivedOffset);
  dTotalOffset = (double)(ClkOffset);
  dTotalOffset += ( ( ( dTimePassed / ( 24.0 * 3600.0 ) )
					     * (double)(ClkDrift) ) / 1000.0 );
  dTimePassed = dTimePassed / ( 24.0 * 3600.0 );
  dTimePassed = dTimePassed * dTimePassed;
  dDrift = dTimePassed * (double)(FreqDrift) / 1000000.0;
  dTotalOffset += ( dDrift / 2.0 );
  return( (int)(dTotalOffset) );

}

/*
 * Execute a command.
 */
static void ProcessCommand( struct ClkClient *Clnt, struct ClkCmd *CmdPnt )
{
  int Len, MicroStep, PpsOutput, Hz, Median, Mode, uSec;
  short sLen;
  FILE *fp;
  struct ppsclockev ClockEv;


  Len = (int)(ntohs( CmdPnt->Length ));
  if( Len < sizeof( short ) || Len > sizeof Clnt->Buffer ) {
    SendResponse( Clnt, GARBAGE_CMD );
    return;
  }
  switch( ntohs( CmdPnt->Type ) ) {

    case CLK_DRIFT:
      ClkOffset = ntohl( CmdPnt->Data.ClkDrift.Offset );
      ClkDrift = ntohl( CmdPnt->Data.ClkDrift.Drift );
      ClkFreqDrift = ntohl( CmdPnt->Data.ClkDrift.FreqDrift );
      LastReceivedOffset = time( NULL );
      if( ( fp = fopen( MicrodConfig.OffsetFile, "w" ) ) != NULL ) {
        fprintf( fp, "%d %d %d %ld\n",
		 ClkOffset, ClkDrift, ClkFreqDrift, LastReceivedOffset );
        fclose( fp );
      }
      SetStepper( ClkOffset );
      syslog( LOG_ERR,
	     "Offset: %d ns, Drift: %d ps/day FreqDrift: %d fs/s/day received",
	     ClkOffset, ClkDrift, ClkFreqDrift );
      SendResponse( Clnt, OK );
      break;

    case CLK_OFFSET:
      ClkOffset = ntohl( CmdPnt->Data.ClkStatus.Offset );
      ClkDrift = ntohl( CmdPnt->Data.ClkStatus.Drift );
      ClkFreqDrift = 0;
      LastReceivedOffset = time( NULL );
      if( ( fp = fopen( MicrodConfig.OffsetFile, "w" ) ) != NULL ) {
        fprintf( fp, "%d %d %d %ld\n",
		 ClkOffset, ClkDrift, ClkFreqDrift, LastReceivedOffset );
        fclose( fp );
      }
      SetStepper( ClkOffset );
      syslog( LOG_ERR, "Offset: %d ns, Drift: %d ps/day received",
	      ClkOffset, ClkDrift );
      SendResponse( Clnt, OK );
      break;

    case CLK_GETTIMESTAMP:
      if( ioctl(PpsFd, CIOGETEV, (caddr_t)&ClockEv) < 0 ) {
        SendResponse( Clnt, IOCTLERROR );
      }
      else {
        uSec = ( ClockEv.tv.tv_usec + 500L ) / 1000L;
	sLen = sizeof( short ) + sizeof( struct ppsclockev );
	CmdPnt->Length = htons( sLen );
	CmdPnt->Type = htons( CLK_TIMESTAMP );
	CmdPnt->Data.ClockEvent.tv.tv_sec = htonl( ClockEv.tv.tv_sec );
	CmdPnt->Data.ClockEvent.tv.tv_usec = htonl( uSec );
	CmdPnt->Data.ClockEvent.serial = htonl( ClockEv.serial );
	MicrodWrite( Clnt, Clnt->Buffer, sLen + sizeof( short ) );
      }
      break;

    case CLK_GETMICROSTEP:
      if( ioctl(PpsFd, CIOGETSTEP, (caddr_t)&MicroStep) < 0 ) {
        SendResponse( Clnt, IOCTLERROR );
      }
      else {
	sLen = sizeof( short ) + sizeof( int );
	CmdPnt->Length = htons( sLen );
	CmdPnt->Type = htons( CLK_MICROSTEP );
	CmdPnt->Data.NanoSec = htonl( MicroStep );
	MicrodWrite( Clnt, Clnt->Buffer, sLen + sizeof( short ) );
      }
      break;

    case CLK_SETMICROSTEP:
      SendResponse( Clnt, INVALID_CMD );
      break;

    case CLK_GETPPSOUTPUT:
      if( ioctl(PpsFd, CIOGETPPS, (caddr_t)&PpsOutput) < 0 ) {
        SendResponse( Clnt, IOCTLERROR );
      }
      else {
	sLen = sizeof( short ) + sizeof( int );
	CmdPnt->Length = htons( sLen );
	CmdPnt->Type = htons( CLK_PPSOUTPUT );
	CmdPnt->Data.NanoSec = htonl( PpsOutput );
	MicrodWrite( Clnt, Clnt->Buffer, sLen + sizeof( short ) );
      }
      break;

    case CLK_SETPPSOUTPUT:
      SendResponse( Clnt, INVALID_CMD );
      break;

    case CLK_GETMED :
      if( ioctl(PpsFd, CIOGETMED, (caddr_t)&Median) < 0 ) {
        SendResponse( Clnt, IOCTLERROR );
      }
      else {
        if( Median < 0 )
          uSec = ( Median - 500 ) / 1000;
        else
          uSec = ( Median + 500 ) / 1000;

	sLen = sizeof( short ) + sizeof( int );
	CmdPnt->Length = htons( sLen );
	CmdPnt->Type = htons( CLK_MED );
	CmdPnt->Data.MicroSec = htonl( uSec );
	MicrodWrite( Clnt, Clnt->Buffer, sLen + sizeof( short ) );
      }
      break;

    case CLK_GETHZ :
      if( ioctl(PpsFd, CIOGETHZ, (caddr_t)&Hz) < 0 ) {
        SendResponse( Clnt, IOCTLERROR );
      }
      else {
	sLen = sizeof( short ) + sizeof( int );
	CmdPnt->Length = htons( sLen );
	CmdPnt->Type = htons( CLK_HZ );
	CmdPnt->Data.NanoSec = htonl( Hz );
	MicrodWrite( Clnt, Clnt->Buffer, sLen + sizeof( short ) );
      }
      break;

    case CLK_GETCLKMODE :
      if( ioctl(PpsFd, CIOGETCLK, (caddr_t)&Mode) < 0 ) {
        SendResponse( Clnt, IOCTLERROR );
      }
      else {
	sLen = sizeof( short ) + sizeof( int );
	CmdPnt->Length = htons( sLen );
	CmdPnt->Type = htons( CLK_CLKMODE );
	CmdPnt->Data.Mode = htonl( Mode );
	MicrodWrite( Clnt, Clnt->Buffer, sLen + sizeof( short ) );
      }
      break;

    default:
      SendResponse( Clnt, INVALID_CMD );
      break;
  }

}

/*
 * Loop through the list of connected clients checking for
 * received data. When a complete command has been received,
 * process the command and return the repsonse.
 */
static void CheckClients( fd_set *Fds )
{
  struct ClkClient *Clnt;
  unsigned char *gPnt;
  int bLen;

NextClient:
  for( Clnt = Clients; Clnt != NULL; Clnt = Clnt->Next ) {
    if( FD_ISSET( Clnt->Sock, Fds ) ) {
      bLen = MicrodRead( Clnt, GlobalBuf, sizeof GlobalBuf );
      if( bLen <= 0 ) {
        RemoveClient( Clnt );
        goto NextClient;
      }
      gPnt = GlobalBuf;
      while( bLen-- > 0 ) {
	switch( Clnt->State ) {
	  case RCV_LENGTH:
	    if( Clnt->BufLength < sizeof( short ) ) {
	      Clnt->Buffer[ Clnt->BufLength++ ] = *gPnt++;
	    }
	    if( Clnt->BufLength >= sizeof( short ) ) {
	      Clnt->ExpectedLength = (int)(ntohs(*((short *)(Clnt->Buffer))));
	      Clnt->ExpectedLength += sizeof( short );
	      Clnt->State = RCV_DATA;
	    }
	    break;

	  case RCV_DATA:
	    if( Clnt->BufLength < Clnt->ExpectedLength ) {
	      Clnt->Buffer[ Clnt->BufLength++ ] = *gPnt++;
	    }
	    if( Clnt->BufLength >= Clnt->ExpectedLength ) {
	      ProcessCommand( Clnt, (struct ClkCmd *)(Clnt->Buffer) );
	      Clnt->State = RCV_LENGTH;
	      Clnt->BufLength = 0;
	    }
	    break;

	  default:
	    Clnt->State = RCV_LENGTH;
	}
      }
    }

  }

}

/*
 * Disconnect the tty and enter daemon mode.
 */
static void DisconnectFromTty()
{
  int i;

  if( fork() ) {
    exit(0);
  }
  i = open("/dev/tty", O_RDWR);
  if( i >= 0 ) {
    (void) ioctl(i, TIOCNOTTY, NULL);
    (void) close(i);
  }

}

/*
 * Main entrypoint.
 * Init everything and then loop receiving data from
 * clients and checking ntp server status.
 */
int main( int argc, char *argv[] )
{
  fd_set Fds;
  int Sock, RxSock, SaLen, i, Line;
  int One = 1;
  time_t Now;
  struct sockaddr_in Sa;
  struct timeval tm;
  FILE *fp;
  double Offset, Dispersion;
  char Buf[ 256 ];

  /*
   * Read configuration file
   */
  bzero( &MicrodConfig, sizeof MicrodConfig );
  MicrodConfig.UseSsl = 1;
  if( argc > 1 )
    strncpy( ConfigFile, argv[ 1 ], sizeof ConfigFile );

  Line = 0;
  if( ( i = ReadConfig( ConfigFile, &Line ) ) != 0 ) {
    if( Line != 0 )
      fprintf( stderr, "Error at line %d in file %s\n", Line, ConfigFile );
    else
      fprintf( stderr, "Error in file %s\n", ConfigFile );

    if( i > MAX_ERROR_NUMBER )
      i = MAX_ERROR_NUMBER;
    fprintf( stderr, "     %s\n", ErrorTexts[ i ] );
    exit( 1 );
  }
  /*
   * If configured for ssl, do all initial ssl configuration.
   */
  if( MicrodConfig.UseSsl ) {
    SSL_load_error_strings();
    SSLeay_add_ssl_algorithms();
    SslMeth = SSLv23_server_method();
    SslCtx = SSL_CTX_new( SslMeth );
    SSL_CTX_set_quiet_shutdown( SslCtx, 1 );
    SSL_CTX_set_options( SslCtx, 0 );
    SSL_CTX_sess_set_cache_size( SslCtx, 8 );
    if( !SslCtx ) {
      fprintf( stderr, "SSL_CTX_NEW: %s\n",
	       ERR_error_string( ERR_get_error(), NULL ) );
      exit(1);
    }
    if( ( !SSL_CTX_load_verify_locations( SslCtx, MicrodConfig.SslCa, NULL ) )
    || ( !SSL_CTX_set_default_verify_paths( SslCtx ) ) ) {
        ERR_print_errors_fp(stderr);
        exit( 1 );
    }
    if( SSL_CTX_use_certificate_file( SslCtx, MicrodConfig.SslCert,
				      SSL_FILETYPE_PEM ) <= 0 ) {
      ERR_print_errors_fp(stderr);
      exit( 1 );
    }
    if( SSL_CTX_use_PrivateKey_file( SslCtx, MicrodConfig.SslKey,
				     SSL_FILETYPE_PEM ) <= 0 ) {
      ERR_print_errors_fp(stderr);
      exit( 1 );
    }
    if( !SSL_CTX_check_private_key( SslCtx ) ) {
      fprintf( stderr,
	       "Private key does not match the certificate public key\n" );
      exit( 1 );
    }
    SSL_CTX_set_tmp_rsa_callback( SslCtx, tmp_rsa_cb );
    SSL_CTX_set_cipher_list( SslCtx, MicrodConfig.SslCipher );
    SSL_CTX_set_verify( SslCtx, ( SSL_VERIFY_PEER
			        | SSL_VERIFY_FAIL_IF_NO_PEER_CERT
			        | SSL_VERIFY_CLIENT_ONCE ), verify_callback );
    SSL_CTX_set_session_id_context( SslCtx, (void*)(&SslSessionIdContext),
 				    sizeof SslSessionIdContext );
    SSL_CTX_set_client_CA_list( SslCtx, SSL_load_client_CA_file( 
							MicrodConfig.SslCa ) );
  }
  /*
   * Redirect some signals.
   */
  (void)signal( SIGTERM, Terminate );
  (void)signal( SIGINT, Terminate );
  (void)signal( SIGQUIT, Terminate );
  (void)signal(SIGPIPE, SIG_IGN );
  /*
   * Try to read present offset/drift from the offset file.
   */
  ClkOffset = ClkDrift = ClkFreqDrift = 0;
  LastReceivedOffset = time( NULL );
  if( ( fp = fopen( MicrodConfig.OffsetFile, "r" ) ) == NULL ) {
    fprintf( stderr,
	     "No offset file found, using 0 as initial offset/drift\n" );
  }
  else {
    if( fread( Buf, 1, sizeof Buf, fp ) <= 0 ) {
      fprintf( stderr,
	       "Can't read offset file, using 0 as initial offset/drift\n" );
    }
    else {
      if( sscanf( Buf, "%d %d %d %ld",
	   &ClkOffset, &ClkDrift, &ClkFreqDrift, &LastReceivedOffset ) != 4 ) {

	if( sscanf( Buf, "%d %d %ld",
		  &ClkOffset, &ClkDrift, &LastReceivedOffset ) != 3 ) {
	  fprintf( stderr,
		   "Syntax error in offset file, using 0 as initial offset/drift/fdrift\n" );
	  ClkOffset = ClkDrift = ClkFreqDrift = 0;
	}
	else {
	  fprintf( stderr,
		 "Old offset file, using offset %d, drift %d, fdrift 0\n",
		 ClkOffset, ClkDrift );
	  ClkFreqDrift = 0;
	}
      }
      else {
	fprintf( stderr,
		 "Using offset %d, drift %d, fdrift %d\n",
		 ClkOffset, ClkDrift, ClkFreqDrift );
      }
    }
    fclose( fp );
  }
  /*
   * Open the pps device.
   */
  if( ( PpsFd = open( MicrodConfig.PpsDevice, O_RDWR ) ) < 0 ) {
    printf( "Pps device %s not found\n", MicrodConfig.PpsDevice );
    exit( 1 );
  }
  if( MicrodConfig.NtpFilter ) {
    /*
     * Create a socket to use for socket ioctl's.
     * These ioctl's turns on/off reception of various types
     * of ntp packets.
     */
    IoctlSock = socket(AF_INET, SOCK_DGRAM, 0);
    if( IoctlSock < 0 ) {
      printf( "Can't create socket for ioctl's\n" );
      exit( 1 );
    }
    /*
     * Create a socket to use when talking to ntpd.
     */
    NtpSock = socket(AF_INET, SOCK_DGRAM, 0);
    if( NtpSock < 0 ) {
      printf( "Can't create socket for ntpd\n" );
      exit( 1 );
    }
    /*
     * Prepare the ntp socket.
     */
    bzero( &Sa, sizeof Sa );
    Sa.sin_len = sizeof( struct sockaddr_in );
    Sa.sin_family = AF_INET;
    Sa.sin_addr.s_addr = inet_addr( "127.0.0.1" );
    Sa.sin_port =  0;
    if( bind( NtpSock, (struct sockaddr *)(&Sa), sizeof Sa ) < 0 ) {
      printf( "Can't bind ntpd socket, errno : %d", errno );
      close( NtpSock );
      close( IoctlSock );
      exit( 1 );
    }
  }
  /*
   * Create a socket for client connections.
   */
  Sock = socket(AF_INET, SOCK_STREAM, 0);
  if (Sock < 0) {
    printf( "Can't create socket, errno : %d\n", errno );
    close( PpsFd );
    exit(1);
  }
  setsockopt( Sock, SOL_SOCKET, SO_REUSEADDR, &One, sizeof One );
  bzero( &Sa, sizeof Sa );
  Sa.sin_len = sizeof( struct sockaddr_in );
  Sa.sin_family = AF_INET;
  Sa.sin_addr.s_addr = INADDR_ANY;
  Sa.sin_port = MicrodConfig.Port;
  if( bind( Sock, (struct sockaddr *)(&Sa), sizeof(Sa) ) < 0 ) {
    printf( "Can't bind socket, errno : %d\n", errno );
    close( PpsFd );
    exit(1);
  }
  listen( Sock, 4 );            /* allow a queue of 4 */
  /*
   * Open syslog and disconnect from the controlling tty.
   */
  openlog( "microd", LOG_PID | LOG_NDELAY, LOG_LOCAL0 );
  DisconnectFromTty();
  if( MicrodConfig.UseSsl )
    syslog( LOG_ERR, "Starting microd in ssl mode" );
  else
    syslog( LOG_ERR, "Starting microd in cleartext mode" );
  /*
   * Do initial setup of microstepper.
   */
  Now = time( NULL );
  SetStepper( CalcOffset( ClkOffset, ClkDrift, ClkFreqDrift,
			  LastReceivedOffset, Now ) );
#if 0
  SetStepper(ClkOffset + 
	      (int)(((double)(Now - LastReceivedOffset) / ( 24.0 * 3600.0 ) )
					  * ((double)(ClkDrift) / 1000.0 ) ) );
#endif
  if( MicrodConfig.NtpFilter ) {
    /*
     * Allow only ntp response packets.
     */
    if( DisallowAllNtp() != 0 )
      syslog( LOG_ERR, "Failed to turn off all ntp reception" );

    if( MicrodConfig.AllowResponses ) {
      if( AllowNtp( NTP_RESPONSE_PKT ) != 0 )
	syslog( LOG_ERR, "Failed to turn on ntp response reception" );
      else
	syslog( LOG_ERR, "Turning on ntp response reception" );
    }
    if( MicrodConfig.AllowControl ) {
      if( AllowNtp( NTP_CONTROL_PKT ) != 0 )
	syslog( LOG_ERR, "Failed to turn on ntp control msg reception" );
      else
	syslog( LOG_ERR, "Turning on ntp control msg reception" );
    }
  }
  /*
   * Loop forever.
   */
  SyncTimer = Now;
  FD_ZERO( &AllFds );
  while( 1 ) {
    /*
     * Time to check ntp server ?
     */
    Now = time( NULL );
    if( MicrodConfig.NtpFilter ) {
      if( Now >= SyncTimer + 60 ) {
        SyncTimer = Now;
        if( GetOffsetAndDispersion( &Offset, &Dispersion ) != 0 ) {
	  syslog( LOG_ERR, "Cant reach ntp server, turning off queries" );
	  if( DisallowNtp( NTP_QUERY_PKT ) != 0 )
	    syslog( LOG_ERR, "Failed to turn off ntp query reception" );

	  Syncronized = 0;
        }
        else {
	  if( Syncronized ) {
	    if( fabs( Offset ) > MicrodConfig.MaxOffset
	    || Dispersion > MicrodConfig.MaxDispersion ) {
	      syslog( LOG_ERR, "Ntp queries is no longer accepted" );
	      if( DisallowNtp( NTP_QUERY_PKT ) != 0 )
	        syslog( LOG_ERR, "Failed to turn off ntp query reception" );

	      Syncronized = 0;
	    }
          }
          else {
	    if( fabs( Offset ) <= MicrodConfig.MaxOffset
	    && Dispersion <= MicrodConfig.MaxDispersion ) {
	      syslog( LOG_ERR, "Ntp queries is now accepted" );
	      if( AllowNtp( NTP_QUERY_PKT ) != 0 )
	        syslog( LOG_ERR, "Failed to turn on ntp query reception" );

	      Syncronized = 1;
	    }
	  }
        }
      }
    }
    if( Now >= LastStepperUpdate + MicrodConfig.StepperInterval ) {
      if( LastReceivedOffset != 0 ) {
	SetStepper( CalcOffset( ClkOffset, ClkDrift, ClkFreqDrift,
				LastReceivedOffset, Now ) );
#if 0
	SetStepper( ClkOffset + 
		(int)(((double)(Now - LastReceivedOffset) / ( 24.0 * 3600.0 ) )
					  * ((double)(ClkDrift) / 1000.0 ) ) );
#endif
      }
    }
    /*
     * Wait for a connection or a timeout.
     */
    tm.tv_sec = 5;
    tm.tv_usec = 0;
    FD_COPY( &AllFds, &Fds );
    FD_SET( Sock, &Fds );
    if( ( i = select( FD_SETSIZE, &Fds, NULL, NULL, &tm ) ) < 0 ) {
      syslog( LOG_NOTICE, "select failed (%m)" );
      close( PpsFd );
      exit( 0 );
    }
    /*
     * Timeout from select.
     */
    if( i == 0 ) {
      continue;
    }
    /*
     * Accept a new connection.
     */
    if( FD_ISSET( Sock, &Fds ) ) {
      SaLen = sizeof Sa;
      RxSock = accept( Sock, (struct sockaddr *)(&Sa), &SaLen );
      if( RxSock < 0 ) {
        syslog( LOG_NOTICE, "Error returned from accept (%m)" );
	close( PpsFd );
      }
      else {
	NewClient( RxSock );
      }
    }
    else {
      /*
       * Data received from already connected client.
       */
      CheckClients( &Fds );
    }

  }
  return 0;
}
