%{
/*
 * Scan configuration file for microd.
 *
 * File : scanner.l
 *
 * Date : 00-01-17
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <openssl/rsa.h>
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/e_os.h>
#include <openssl/buffer.h>
#include <openssl/bio.h>

#include "ppsclock.h"
#include "microd.h"

static int IntervalFound = 0;
static int PortFound = 0;
static int DevaddrFound = 0;
static int MaxoffFound = 0;
static int MaxdispFound = 0;
static int PpsdevFound = 0;
static int OfileFound = 0;
static int NtpfilterFound = 0;
static int AllowrespFound = 0;
static int AllowcontrolFound = 0;

static int CertFound = 0;
static int KeyFound = 0;
static int CaFound = 0;
static int CipherFound = 0;
static int UsesslFound = 0;

%}

%s INTERVAL PORT DEVADDR MAXOFF MAXDISP PPSDEV OFILE CERTIF KEY CA CIPHER USESSL NTPFILTER ALLOWRESP ALLOWCONTROL

%%

[ \;\t\n]+		 ;
#+.*\n			 ;
<INITIAL>updateinterval|interval {
	  BEGIN(INTERVAL);
	}
<INITIAL>clientport|port {
	  BEGIN(PORT);
	}
<INITIAL>deviceaddress|devaddr {
	  BEGIN(DEVADDR);
	}
<INITIAL>maxoffset|maxoff {
	  BEGIN(MAXOFF);
	}
<INITIAL>maxdispersion|maxdisp {
	  BEGIN(MAXDISP);
	}
<INITIAL>ppsdevice|ppsdev {
	  BEGIN(PPSDEV);
	}
<INITIAL>offsetfile|ofile {
	  BEGIN(OFILE);
	}
<INITIAL>certfile|cert {
	  BEGIN(CERTIF);
	}
<INITIAL>keyfile|key {
	  BEGIN(KEY);
	}
<INITIAL>cafile|ca {
	  BEGIN(CA);
	}
<INITIAL>cipher {
	  BEGIN(CIPHER);
	}
<INITIAL>usessl {
	  BEGIN(USESSL);
	}
<INITIAL>ntpfilter {
	  BEGIN(NTPFILTER);
	}
<INITIAL>allowresponses|allowresp {
	  BEGIN(ALLOWRESP);
	}
<INITIAL>allowcontrol|allowctrl {
	  BEGIN(ALLOWCONTROL);
	}


<INTERVAL>[0-9]+	{
	  int i;

	  if( sscanf( yytext, "%d", &i ) != 1 )
	    return( CONFIG_ERROR );

	  if( i < 1 || i > ( 24 * 3600 ) )
	    return( INVALID_INTERVAL );

	  MicrodConfig.StepperInterval = i;
	  IntervalFound = 1;
	  BEGIN(INITIAL);
	}
<PORT>[0-9]+	{
	  int i;

	  if( sscanf( yytext, "%d", &i ) != 1 )
	    return( CONFIG_ERROR );

	  if( i < 1 || i > 65535 )
	    return( INVALID_PORT );

	  MicrodConfig.Port = htons( (unsigned short)(i) );
	  PortFound = 1;
	  BEGIN(INITIAL);
	}
<DEVADDR>[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}	{
	  MicrodConfig.DeviceAddress = inet_addr( yytext );
	  DevaddrFound = 1;
	  BEGIN(INITIAL);
	}
<MAXOFF>[0-9]\.[0-9]+	{
	  double d;

	  if( sscanf( yytext, "%lf", &d ) != 1 )
	    return( CONFIG_ERROR );

	  if( d < 0.0 || d > 0.999999 )
	    return( INVALID_MAXOFFSET );

	  MicrodConfig.MaxOffset = d;
	  MaxoffFound = 1;
	  BEGIN(INITIAL);
	}
<MAXDISP>[0-9]\.[0-9]+	{
	  double d;

	  if( sscanf( yytext, "%lf", &d ) != 1 )
	    return( CONFIG_ERROR );

	  if( d < 0.0 || d > 0.999999 )
	    return( INVALID_MAXDISPERSION );

	  MicrodConfig.MaxDispersion = d;
	  MaxdispFound = 1;
	  BEGIN(INITIAL);
	}
<PPSDEV>(\/[a-z0-9_\.\-]+)+	{
	  strncpy( MicrodConfig.PpsDevice, yytext,
		   sizeof MicrodConfig.PpsDevice );
	  PpsdevFound = 1;
	  BEGIN(INITIAL);
	}
<OFILE>(\/[a-z0-9_\.\-]+)+	{
	  strncpy( MicrodConfig.OffsetFile, yytext,
		   sizeof MicrodConfig.OffsetFile );
	  OfileFound = 1;
	  BEGIN(INITIAL);
	}
<CERTIF>(\/[a-z0-9_\.\-]+)+	{
	  strncpy( MicrodConfig.SslCert, yytext,
		   sizeof MicrodConfig.SslCert );
	  CertFound = 1;
	  BEGIN(INITIAL);
	}
<KEY>(\/[a-z0-9_\.\-]+)+	{
	  strncpy( MicrodConfig.SslKey, yytext,
		   sizeof MicrodConfig.SslKey );
	  KeyFound = 1;
	  BEGIN(INITIAL);
	}
<CA>(\/[a-z0-9_\.\-]+)+	{
	  strncpy( MicrodConfig.SslCa, yytext,
		   sizeof MicrodConfig.SslCa );
	  CaFound = 1;
	  BEGIN(INITIAL);
	}
<CIPHER>([a-z0-9_\-]+)+	{
	  strncpy( MicrodConfig.SslCipher, yytext,
		   sizeof MicrodConfig.SslCipher );
	  CipherFound = 1;
	  BEGIN(INITIAL);
	}
<USESSL>true|yes	{
	  MicrodConfig.UseSsl = 1;
	  UsesslFound = 1;
	  BEGIN(INITIAL);
	}
<USESSL>false|no	{
	  MicrodConfig.UseSsl = 0;
	  UsesslFound = 1;
	  BEGIN(INITIAL);
	}
<NTPFILTER>true|yes	{
	  MicrodConfig.NtpFilter = 1;
	  NtpfilterFound = 1;
	  BEGIN(INITIAL);
	}
<NTPFILTER>false|no	{
	  MicrodConfig.NtpFilter = 0;
	  NtpfilterFound = 1;
	  BEGIN(INITIAL);
	}
<ALLOWRESP>true|yes	{
	  MicrodConfig.AllowResponses = 1;
	  AllowrespFound = 1;
	  BEGIN(INITIAL);
	}
<ALLOWRESP>false|no	{
	  MicrodConfig.AllowResponses = 0;
	  AllowrespFound = 1;
	  BEGIN(INITIAL);
	}
<ALLOWCONTROL>true|yes	{
	  MicrodConfig.AllowControl = 1;
	  AllowcontrolFound = 1;
	  BEGIN(INITIAL);
	}
<ALLOWCONTROL>false|no	{
	  MicrodConfig.AllowControl = 0;
	  AllowcontrolFound = 1;
	  BEGIN(INITIAL);
	}
.	{
	  return( CONFIG_ERROR );
	}

%%

int ReadConfig( char *ConfigFile, int *LineNo )
{
  int RetCode;

  if( ( yyin = fopen( ConfigFile, "r" ) ) == NULL )
    return( NO_CONFIG_FILE );

  *LineNo = 0;
  RetCode = yylex();
  fclose( yyin );
  if( RetCode == CONFIG_ERROR ) {
    *LineNo = yylineno;
    return( RetCode );
  }
  if( !IntervalFound )
    return( NO_INTERVAL );

  if( !PortFound )
    return( NO_PORT );

  if( !PpsdevFound )
    return( NO_PPSDEVICE );

  if( !OfileFound )
    return( NO_OFFSETFILE );

  if( !UsesslFound )
    return( NO_USESSL );

  if( !NtpfilterFound )
    return( NO_NTPFILTER );

  if( MicrodConfig.NtpFilter ) {
    if( !DevaddrFound )
      return( NO_DEVADDR );

    if( !MaxoffFound )
      return( NO_MAXOFFSET );

    if( !MaxdispFound )
      return( NO_MAXDISPERSION );

    if( !AllowrespFound )
      return( NO_ALLOWRESP );

    if( !AllowcontrolFound )
      return( NO_ALLOWCONTROL );
  }
  if( MicrodConfig.UseSsl ) {
    if( !CertFound )
      return( NO_CERTFILE );

    if( !KeyFound )
      return( NO_KEYFILE );

    if( !CaFound )
      return( NO_CAFILE );

    if( !CipherFound )
      return( NO_CIPHER );
  }
  *LineNo = yylineno;
  return( RetCode );

}
