/*
 * File : setoffset.c
 *
 * Transfer new offset and drift to microstepper daemon.
 * Use ssl or cleartext communication.
 *
 * Date : 000129
 */

#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <signal.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/file.h>
#include <netinet/in.h>
#include <netdb.h>

#include <openssl/rsa.h>
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/e_os.h>
#include <openssl/buffer.h>
#include <openssl/bio.h>

#include "ppsclock.h"
#include "microd.h"

void Usage( char *Prog )
{

  printf( "Usage: %s [ -ssl ] hostname offset drift\n", Prog );
  exit( 1 );

}


/*
 * Main entrypoint.
 */
int main(argc, argv)
  int argc;
  char *argv[];
{
  char* certfile="/etc/certificate/Microd_cert.pem";
  char* keyfile="/etc/certificate/Microd_key.pem";
  char* cafile="/etc/certificate/Cacert.pem";
  char* auth="RC4-MD5";
  SSL_CTX* ctx = NULL;
  SSL_METHOD *meth = NULL;
  SSL *ssl = NULL;
  int Sock, Len, Offset, Drift, Arg, UseSsl;
  char HostName[ 128 ], Buf[ 1024 ];
  struct ClkCmd *CmdPnt;
  struct sockaddr_in Sa;
  struct hostent *Host;


  /*
   * Check if SSL communication is requested.
   */
  if( argc > 1 && strcasecmp( argv[ 1 ], "-ssl" ) == 0 ) {
    UseSsl = 1;
    Arg = 2;
  }
  else {
    UseSsl = 0;
    Arg = 1;
  }
  /*
   * Scan all remaining command line arguments.
   */
  if( argc < Arg + 3 ) {
    Usage( argv[ 0 ] );
  }
  if( sscanf( argv[ Arg ], "%s", HostName ) != 1 ) {
    Usage( argv[ 0 ] );
  }
  if( sscanf( argv[ Arg + 1 ], "%d", &Offset ) != 1 ) {
    Usage( argv[ 0 ] );
  }
  if( sscanf( argv[ Arg + 2 ], "%d", &Drift ) != 1 ) {
    Usage( argv[ 0 ] );
  }
  /*
   * Try to resolve server hostname.
   */
  if( ( Host = gethostbyname( HostName ) ) == NULL ) {
    printf( "Can't resolve server hostname %s", HostName );
    exit( 1 );
  }
  /*
   * If SSL is requested, init ssl.
   */
  if( UseSsl ) {
    SSL_load_error_strings();
    SSLeay_add_ssl_algorithms();
    meth = SSLv23_client_method();
    ctx = SSL_CTX_new( meth );

    if( !ctx ) {
      syslog( LOG_ERR, "SSL_CTX_NEW: %s",
	      ERR_error_string( ERR_get_error(), NULL ) );
      exit( 2 );
    }
    SSL_CTX_set_options( ctx, 0 );
    if( SSL_CTX_use_certificate_file( ctx, certfile,
				      SSL_FILETYPE_PEM ) <= 0 ) {
      ERR_print_errors_fp( stderr );
      exit( 3 );
    }
    if( SSL_CTX_use_PrivateKey_file( ctx, keyfile, SSL_FILETYPE_PEM ) <= 0 ) {
      ERR_print_errors_fp( stderr );
      exit( 4 );
    }
    SSL_CTX_set_cipher_list( ctx, auth );
    if( ( !SSL_CTX_load_verify_locations( ctx, cafile, NULL ) )
    || ( !SSL_CTX_set_default_verify_paths( ctx ) ) ) {
      ERR_print_errors_fp( stderr );
      exit( 6 );
    }
    ssl = SSL_new( ctx );
  }
  /*
   * Create a stream socket.
   */
  Sock = socket(AF_INET, SOCK_STREAM, 0);
  if( Sock < 0 ) {
    printf( "Can't create socket" );
    exit( 1 );
  }
  /*
   * Bind the socket.
   */
  bzero( &Sa, sizeof Sa );
  Sa.sin_family = AF_INET;
  Sa.sin_port =  htons( MICROD_PORT );
  Sa.sin_addr.s_addr = *((u_long *)(Host->h_addr));
  if( connect( Sock, (struct sockaddr *)(&Sa), sizeof(Sa) ) < 0 ) {
    printf( "Can't connect to server" );
    close( Sock );
    exit(1);
  }
  /*
   * If SSL is requested, do a SSL-connect.
   */
  if( UseSsl ) {
    SSL_set_fd( ssl, Sock );
    SSL_connect( ssl );
    printf( "Connected to microd using ssl\n" );
  }
  else
    printf( "Connected to microd in clear text\n" );

  /*
   * Format a offset command in the buffer.
   */
  CmdPnt = (struct ClkCmd *)(Buf);
  Len = sizeof( short ) + sizeof( struct ClockData );
  CmdPnt->Length = htons( Len );
  CmdPnt->Type = htons( CLK_OFFSET );
  CmdPnt->Data.ClkStatus.Offset = htonl( Offset );
  CmdPnt->Data.ClkStatus.Drift = htonl( Drift );
  CmdPnt->Data.ClkStatus.Leap = htons( 0 );
  CmdPnt->Data.ClkStatus.Status = htons( 0 );
  /*
   * Write the command in SSL mode if requested.
   * Do a conventional socket write if SSL is not used.
   * Wait for a response from the server.
   */
  if( UseSsl ) {
    SSL_write( ssl, Buf, Len + sizeof( short ) );
    printf( "Data written\n" );
    if( SSL_read( ssl, Buf, 3 * sizeof( short ) ) != 3 * sizeof( short ) ) {
      printf( "Ssl read error\n" );
      SSL_shutdown( ssl );
      shutdown( SSL_get_fd( ssl ), 2 );
      close( Sock );
      if( ssl != NULL )
        SSL_free( ssl );

      if( ctx != NULL )
        SSL_CTX_free( ctx );

      exit( 1 );
    }
  }
  else {
    write( Sock, Buf, Len + sizeof( short ) );
    printf( "Data written\n" );
    if( read( Sock, Buf, 3 * sizeof( short ) ) != 3 * sizeof( short ) ) {
      printf( "Read error\n" );
      close( Sock );
      exit( 1 );
    }
  }
  /*
   * Check the response.
   */
  if( ntohs( CmdPnt->Type ) == CLK_ECODE ) {
    if( ntohs( CmdPnt->Data.Ecode ) == OK )
      printf( "OK returned from microd\n" );
    else
      printf( "%d returned from microd\n", ntohs( CmdPnt->Data.Ecode ) );
  }
  else {
    printf( "A block type %d was returned from microd\n",
	    ntohs( CmdPnt->Type ) );
  }
  /*
   * If SSL is requested, shutdown ssl.
   */
  if( UseSsl ) {
    SSL_shutdown( ssl );
    shutdown( SSL_get_fd( ssl ), 2 );
    close( Sock );
    if( ssl != NULL )
      SSL_free( ssl );

    if( ctx != NULL )
      SSL_CTX_free( ctx );
  }
  else
    close( Sock );

  return 0;

}
