#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/param.h>
#include <gtk/gtk.h>

#include <openssl/rsa.h>
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/e_os.h>
#include <openssl/buffer.h>
#include <openssl/bio.h>

#include "ppsclock.h"
#include "microd.h"

#define N_SAMPLES 500
#define BORDERWIDTH 6

gint SetScaleHandler( GtkWidget *widget );

struct ScaleEntry { int		Value;
		    char	*MaxText;
		    char	*ZeroText;
		    char	*MinText;
};

struct ScaleEntry ScaleDef[] =
{
  {      50, "   50", "    0", "  -50" },
  {     100, "  100", "    0", " -100" },
  {     250, "  250", "    0", " -250" },
  {     500, "  500", "    0", " -500" },
  {    1000, " 1000", "    0", "-1000" },
  {    2500, " 2500", "    0", "-2500" },
  {    5000, " 5000", "    0", "-5000" },
  {   10000, "  10K", "    0", " -10K" },
  {   25000, "  25K", "    0", " -25K" },
  {   50000, "  50K", "    0", " -50K" },
  {  100000, " 100K", "    0", "-100K" },
  {  250000, " 250K", "    0", "-250K" },
  {  500000, " 500K", "    0", "-500K" },
  { 1000000, "   1M", "    0", "  -1M" },
  { -1, "", "", "" }
};

struct MenuEntry { char		*Text;
		   int		(*Handler)( GtkWidget * );
		   int		Selection;
};

struct MenuEntry DrawMenues[] =
{
  { " Auto ", SetScaleHandler, 99 },
  { " 50 ", SetScaleHandler, 0 },
  { " 100 ", SetScaleHandler, 1 },
  { " 250 ", SetScaleHandler, 2 },
  { " 500 ", SetScaleHandler, 3 },
  { " 1000 ", SetScaleHandler, 4 },
  { " 2500 ", SetScaleHandler, 5 },
  { " 5000 ", SetScaleHandler, 6 },
  { " 10000 ", SetScaleHandler, 7 },
  { " 25000 ", SetScaleHandler, 8 },
  { " 50000 ", SetScaleHandler, 9 },
  { " 100000 ", SetScaleHandler, 10 },
  { " 250000 ", SetScaleHandler, 11 },
  { " 500000 ", SetScaleHandler, 12 },
  { " 1000000 ", SetScaleHandler, 13 },
  { "", NULL, -1 }
};


GtkWidget *Widget = NULL;
GdkPixmap *Pixmap = NULL;
GdkGC *LineGc = NULL;
GtkWidget *DrawMenu = NULL;
GtkWidget *RateWindow = NULL;
GtkWidget *StatsWindow = NULL;
GtkWidget *OffsetLabel = NULL;
GtkWidget *StepLabel = NULL;
GtkWidget *PpsLabel = NULL;
GtkWidget *FreqLabel = NULL;
GtkWidget *ClkModeLabel = NULL;

int PresentScale = 0;
int LineWidth = 1;
int SampleMin = 9999999;
int SampleMax = -9999999;
int ScaleMax = 50;
int ScaleMin = -50;
int AutoScale = 1;
int Delay = 30;
int DelayCounter = 30;
int UseSsl = 0;
char SslCertFile[ 256 ] = "/etc/certificate/Microd_cert.pem";
char SslKeyFile[ 256 ] = "/etc/certificate/Microd_key.pem";
char SslCaFile[ 256 ] = "/etc/certificate/Cacert.pem";
char SslCipher[ 256 ] = "RC4-MD5";
SSL_CTX *SslCtx = NULL;
SSL_METHOD *SslMethod = NULL;
SSL *Ssl = NULL;
double PixelsPerSample = 1.0;
double LinesPerUnit = 1.0;
char Host[ 128 ] = "Zoo.Everyware.SE";
int ReadFd = -1;
int nRead = 0;
int State = 0;
int Length = 0;
int nSamples = 0;
int SavedSamples[ N_SAMPLES ];
unsigned char gBuf[ 1024 ];
unsigned char bBuf[ 16384 ];

int InitSsl( char *CertFile, char *KeyFile, char *CaFile, char *Cipher )
{

  if( CertFile != NULL )
    strncpy( SslCertFile, CertFile, sizeof SslCertFile );

  if( KeyFile != NULL )
    strncpy( SslKeyFile, KeyFile, sizeof SslKeyFile );

  if( CaFile != NULL )
    strncpy( SslCaFile, CaFile, sizeof SslCaFile );

  if( Cipher != NULL )
    strncpy( SslCipher, Cipher, sizeof SslCipher );

  SSL_load_error_strings();
  SSLeay_add_ssl_algorithms();
  SslMethod = SSLv23_client_method();
  SslCtx = SSL_CTX_new( SslMethod );
  if( SslCtx == NULL )
    return( -1 );

  SSL_CTX_set_options( SslCtx, 0 );
  if( SSL_CTX_use_certificate_file( SslCtx, SslCertFile,
				    SSL_FILETYPE_PEM ) <= 0 ) {
    SSL_CTX_free( SslCtx );
    SslCtx = NULL;
    return( -1 );
  }
  if( SSL_CTX_use_PrivateKey_file( SslCtx, SslKeyFile,
				   SSL_FILETYPE_PEM ) <= 0 ) {
    SSL_CTX_free( SslCtx );
    SslCtx = NULL;
    return( -1 );
  }
  SSL_CTX_set_cipher_list( SslCtx, SslCipher );
  if( ( !SSL_CTX_load_verify_locations( SslCtx, SslCaFile, NULL ) )
  || ( !SSL_CTX_set_default_verify_paths( SslCtx ) ) ) {
    SSL_CTX_free( SslCtx );
    SslCtx = NULL;
    return( -1 );
  }
  return( 0 );

}


/*
 * Wrapped socket write.
 * Select write or SSL_write according to MicrodConfig.UseSsl.
 */
static int MicrodWrite( void *Buf, int Length )
{
  int RetCode;

  if( UseSsl ) {
try_again:
    RetCode = SSL_write( Ssl, Buf, Length );
    if( RetCode < 0 ) {
      RetCode = SSL_get_error( Ssl, RetCode );
      if( RetCode == SSL_ERROR_WANT_WRITE ) {
        sleep( 1 );
        goto try_again;
      }
    }
  }
  else {
    RetCode = write( ReadFd, Buf, Length );
  }
  return( RetCode );
}

/*
 * Wrapped socket read.
 * Select read or SSL_read according to MicrodConfig.UseSsl.
 */
static int MicrodRead( void *Buf, int Length )
{
  int RetCode;

  if( UseSsl ) {
    for( ; ; ) {
      RetCode = SSL_read( Ssl, Buf, Length );
      switch( SSL_get_error( Ssl, RetCode ) ) {
        case SSL_ERROR_NONE:
          return( RetCode );

        case SSL_ERROR_WANT_WRITE:
        case SSL_ERROR_WANT_READ:
        case SSL_ERROR_WANT_X509_LOOKUP:
          sleep( 1 );
          break;
        case SSL_ERROR_SYSCALL:
        case SSL_ERROR_SSL:
          return( -1 );
        case SSL_ERROR_ZERO_RETURN:
          return( 0 );

        default:
          sleep( 1 );
      }
    }
  }
  else {
    RetCode = read( ReadFd, Buf, Length );
  }
  return( RetCode );
}

/*
 * Connect to server.
 */
int MicrodConnect( char *HostName )
{
  struct hostent *Host;
  struct sockaddr_in Sa;
  int Sock;

  if( ( Host = gethostbyname( HostName ) ) == NULL ) {
    printf( "Can't resolve server hostname %s", HostName );
    exit( 1 );
  }
  Sock = socket(AF_INET, SOCK_STREAM, 0);
  if( Sock < 0 ) {
    printf( "Can't create socket" );
    return( -1 );
  }
  /*
   * Bind the socket.
   */
  bzero( &Sa, sizeof Sa );
  Sa.sin_family = AF_INET;
  Sa.sin_port =  htons( MICROD_PORT );
  Sa.sin_addr.s_addr = *((u_long *)(Host->h_addr));
  if( connect( Sock, (struct sockaddr *)(&Sa), sizeof(Sa) ) < 0 ) {
    printf( "Can't connect to server" );
    close( Sock );
    return( -1 );
  }
  return( Sock );

}


/*
 * Plot a single value by drawing a line from the previous
 * value. Update the supplied rectangle struct with the
 * coordinates of the drawn area so that the caller can
 * generate expose events to update the screen.
 */
void DrawSample( int Sample, GdkRectangle *Rect )
{
  int sx, sy, ex, ey, PresVal, PrevVal, PrevSample;
  int Max1, Max2, Min1, Min2;

  Max1 = Max2 = Min1 = Min2 = 0;
  if( Sample > 0 )
    PrevSample = Sample - 1;
  else
    PrevSample = Sample;

  PresVal = SavedSamples[ Sample ];
  if( PresVal > ScaleMax ) {
    PresVal = ScaleMax;
    Max2 = 1;
  }
  else if( PresVal < ScaleMin ) {
    PresVal = ScaleMin;
    Min2 = 1;
  }
  PrevVal = SavedSamples[ PrevSample ];
  if( PrevVal > ScaleMax ) {
    PrevVal = ScaleMax;
    Max1 = 1;
  }
  else if( PrevVal < ScaleMin ) {
    PrevVal = ScaleMin;
    Min1 = 1;
  }
  if( ( Max1 && Max2 ) || ( Min1 && Min2 ) )
    return;

  sx = (int)((double)(PrevSample) * PixelsPerSample + 0.5 ) + BORDERWIDTH * 6;
  sy = (int)((double)(ScaleMax - PrevVal) * LinesPerUnit + 0.5 ) + BORDERWIDTH;
  ex = (int)((double)(Sample) * PixelsPerSample + 0.5 ) + BORDERWIDTH * 6;
  ey = (int)((double)(ScaleMax - PresVal) * LinesPerUnit + 0.5 ) + BORDERWIDTH;
  gdk_draw_line( Pixmap, LineGc, sx, sy, ex, ey );
  if( sx < ex ) {
    Rect->x = sx - ( LineWidth / 2 );
    Rect->width = ( ex - sx ) + 1 + LineWidth;
  }
  else {
    Rect->x = ex - ( LineWidth / 2 );
    Rect->width = ( sx - ex ) + 1 + LineWidth;
  }
  if( sy < ey ) {
    Rect->y = sy - ( LineWidth / 2 );
    Rect->height = ( ey - sy ) + 1 + LineWidth;
  }
  else {
    Rect->y = ey - ( LineWidth / 2 );
    Rect->height = ( sy - ey ) + 1 + LineWidth;
  }
#if 0
  printf( "No:%d Prev:%d Pres:%d Sx:%d Sy:%d Ex:%d Ey:%d\n",
	  Sample, PrevVal, PresVal, sx, sy, ex, ey );
#endif
}

/*
 * Plot a single sample and generate expose events to
 * update the screen.
 */
void UpdateSample( int Sample )
{
  GdkRectangle update_rect;

  DrawSample( Sample, &update_rect );
  gtk_widget_draw( Widget, &update_rect );

}

/*
 * Redraw the entire plot window.
 */
void RedrawSamples()
{
  int i;
  GdkRectangle update_rect;
  int Width = Widget->allocation.width;
  int Height = Widget->allocation.height;
  int sx = BORDERWIDTH * 6;
  int sy = BORDERWIDTH;
  int ex = Widget->allocation.width - BORDERWIDTH;
  int ey = Widget->allocation.height - BORDERWIDTH;
  GdkFont *Font;

  if( !Widget )
    return;

  /*
   * Fill the pixmap with white background.
   */
  gdk_draw_rectangle( Pixmap, Widget->style->white_gc,
		      TRUE, 0, 0, Width, Height );

  /*
   * Draw a frame and horizontal and vertical black lines.
   */
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 sx, sy, ex - 1, sy );
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 sx, ey - 1, ex - 1, ey - 1 );
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 sx, sy, sx, ey - 1 );
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 ex - 1, sy, ex - 1, ey - 1 );

  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 sx, ( ( ey - sy ) / 4 ) + sy,
		 ex - 1, ( ( ey - sy ) / 4 ) + sy );
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 sx, ( ( ey - sy ) / 2 ) + sy,
		 ex - 1, ( ( ey - sy ) / 2 ) + sy );
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 sx, ( 3 * ( ey - sy ) / 4 ) + sy,
		 ex - 1, ( 3 * ( ey - sy ) / 4 ) + sy );

  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 ( ( ex - sx ) / 4 ) + sx, sy,
		 ( ( ex - sx ) / 4 ) + sx, ey - 1 );
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 ( ( ex - sx ) / 2 ) + sx, sy,
		 ( ( ex -sx ) / 2 ) + sx, ey - 1 );
  gdk_draw_line( Pixmap, Widget->style->black_gc,
		 ( ( 3 * ( ex - sx ) ) / 4 ) + sx, sy,
		 ( ( 3 * ( ex -sx ) ) / 4 ) + sx, ey - 1 );

  Font = gdk_font_load( "fixed" );
  gdk_draw_string( Pixmap, Font, Widget->style->black_gc, 2, sy + 6,
		   ScaleDef[ PresentScale ].MaxText );
  gdk_draw_string( Pixmap, Font, Widget->style->black_gc, 2,
		   ( ( ey - sy ) / 2 ) + 10,
		   ScaleDef[ PresentScale ].ZeroText );
  gdk_draw_string( Pixmap, Font, Widget->style->black_gc, 2, ey + 2,
		   ScaleDef[ PresentScale ].MinText );

  /*
   * Plot all samples.
   */
  for( i = 0; i < nSamples; i++ )
    DrawSample( i, &update_rect );

  /*
   * Generate a expose event for the entire pixmap.
   */
  update_rect.x = 0;
  update_rect.y = 0;
  update_rect.width = Width;
  update_rect.height = Height;
  gtk_widget_draw( Widget, &update_rect );

}

/*
 * Send a command code to the server.
 */
void SendCommand( int Cmd )
{
  struct ClkCmd *CmdPnt;

  CmdPnt = (struct ClkCmd *)(gBuf);
  CmdPnt->Length = htons( 2 * sizeof( short ) );
  CmdPnt->Type = htons( Cmd );
  CmdPnt->Data.Ecode = htons( OK );
  MicrodWrite( gBuf, sizeof( short ) + 2 * sizeof( short ) );

}

/*
 * Signal handler.
 */
void Terminate( int sig )
{
#if 0
  printf( "Terminating\n" );
#endif
  close( ReadFd );
  gtk_main_quit();

}

/*
 * Read and decode data arriving from the server.
 */
void ReadMicrod( gpointer Data, gint Source, GdkInputCondition Condition )
{
  int bLen, i, j;
  struct ClkCmd *CmdPnt;
  long Diff;
  char String[ 256 ], *bPnt;


  bLen = MicrodRead( bBuf, sizeof bBuf );
  if( bLen <= 0 ) {
    printf( "Read error Length : %d\n", bLen );
    if( UseSsl ) {
#if 0
      SSL_shutdown( Ssl );
      shutdown( SSL_get_fd( Ssl ), 2 );
#endif
      close( ReadFd );
      if( Ssl != NULL )
	SSL_free( Ssl );

      if( SslCtx != NULL )
	SSL_CTX_free( SslCtx );
    }
    else
      close( ReadFd );

    gtk_main_quit();
  }
  bPnt = bBuf;
  while( bLen-- > 0 ) {
    switch( State ) {

      /*
       * Read the initial length integer.
       */
      case 0 :
	gBuf[ nRead++ ] = *bPnt++;
        if( nRead >= sizeof( short ) ) {
	  nRead = 0;
	  State = 1;
	  Length = (int)(ntohs( *((short *)(gBuf)) ));
	}
	break;

      /*
       * Read the rest of the data according to the length integer.
       */
      case 1 :
	gBuf[ nRead + sizeof( short ) ] = *bPnt++;
	nRead++;
	/*
	 * Complete datablock read ?
	 */
	if( nRead >= Length ) {
	  nRead = 0;
	  State = 0;
	  CmdPnt = (struct ClkCmd *)(gBuf);
	  switch( ntohs( CmdPnt->Type ) ) {

	    case CLK_ECODE :
	      if( ntohs( CmdPnt->Data.Ecode ) != OK ) {
		printf( "Error %d received from server\n",
			ntohs( CmdPnt->Data.Ecode ) );
	      }
	      break;

	    case CLK_MED :
	      Diff = ntohl( CmdPnt->Data.MicroSec );
	      if( StatsWindow != NULL ) {
		sprintf( String, "%ld", Diff );
		gtk_label_set( (GtkLabel *)(OffsetLabel), String );
	      }
	      if( nSamples >= N_SAMPLES ) {
		SampleMin = 9999999;
		SampleMax = -9999999;
		for( i = 0, j = N_SAMPLES / 4; j < N_SAMPLES; i++, j++ ) {
		  SavedSamples[ i ] = SavedSamples[ j ];
		  if( abs( SavedSamples[ j ] ) > SampleMax )
		    SampleMax = abs( SavedSamples[ j ] );
		}
		nSamples = i;
		if( abs( (int)(Diff) ) > SampleMax )
		  SampleMax = abs( (int)(Diff) );

		SavedSamples[ nSamples++ ] = (int)(Diff);
		for( i = 0; ScaleDef[ i ].Value < SampleMax; i++ )
		  if( ScaleDef[ i ].Value < 0 )
		    break;

		if( i != PresentScale && AutoScale ) {
		  ScaleMin = -ScaleDef[ i ].Value;
		  ScaleMax = ScaleDef[ i ].Value;
		  PresentScale = i;
		  LinesPerUnit = (double)(Widget->allocation.height -
					  ( 2 * BORDERWIDTH ) ) /
		    (double)(ScaleMax - ScaleMin);
		}
		RedrawSamples();
	      }
	      else {
		if( abs( (int)(Diff) ) > SampleMax )
		  SampleMax = abs( (int)(Diff) );

		SavedSamples[ nSamples ] = (int)(Diff);
		for( i = 0; ScaleDef[ i ].Value < SampleMax; i++ )
		  if( ScaleDef[ i ].Value < 0 )
		    break;

		if( i != PresentScale && AutoScale ) {
		  ScaleMin = -ScaleDef[ i ].Value;
		  ScaleMax = ScaleDef[ i ].Value;
		  PresentScale = i;
		  LinesPerUnit = (double)(Widget->allocation.height -
					  ( 2 * BORDERWIDTH ) ) /
		    (double)(ScaleMax - ScaleMin);
		  nSamples++;
		  RedrawSamples();
		}
		else
		  UpdateSample( nSamples++ );
	      }
	      break;

	    case CLK_MICROSTEP :
	      if( StatsWindow != NULL ) {
		sprintf( String, "%ld", ntohl( CmdPnt->Data.NanoSec ) );
		gtk_label_set( (GtkLabel *)(StepLabel), String );
	      }
	      break;

	    case CLK_PPSOUTPUT :
	      if( StatsWindow != NULL ) {
		sprintf( String, "%ld", ntohl( CmdPnt->Data.NanoSec ) );
		gtk_label_set( (GtkLabel *)(PpsLabel), String );
	      }
	      break;

	    case CLK_HZ :
	      if( StatsWindow != NULL ) {
		sprintf( String, "%ld", ntohl( CmdPnt->Data.Hz ) );
		gtk_label_set( (GtkLabel *)(FreqLabel), String );
	      }
	      break;

	    case CLK_CLKMODE :
	      if( StatsWindow != NULL ) {
		if( CmdPnt->Data.Mode != 0 )
		  gtk_label_set( (GtkLabel *)(ClkModeLabel), "Internal" );
		else
		  gtk_label_set( (GtkLabel *)(ClkModeLabel), "External" );
	      }
	      break;

	    default :
	      break;
	  }
	}
	break;

      default :
	State = 0;
    }
  }

}

/*
 * Timer routine initiating a command exchange with the server.
 * Called once every second.
 */
gint TickTock( gpointer Data )
{

  if( --DelayCounter <= 0 ) {
    if( State != 0 ) {
      printf( "Previous SendCommand failed\n" );
    }
    SendCommand( CLK_GETMED );
    if( StatsWindow ) {
      SendCommand( CLK_GETMICROSTEP );
      SendCommand( CLK_GETPPSOUTPUT );
      SendCommand( CLK_GETHZ );
      SendCommand( CLK_GETCLKMODE );
    }
    DelayCounter = Delay;
  }
  return( TRUE );

}


/*
 * Stats window delete event callback.
 */
gint StatExit( GtkWidget *widget, GdkEvent *event, gpointer data )
{
  gtk_widget_destroy( widget );
  StatsWindow = NULL;
  return( TRUE );
}


/*
 * Stats menu handler.
 */
gint StatsHandler( GtkWidget *widget, GdkEventButton *event )
{
  GtkWidget *window;
  GtkWidget *table;
  GtkWidget *button, *temp;
  char Title[ 256 ];

  if( StatsWindow )
    return( TRUE );

  window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
  strcpy( Title, "Stats: " );
  strcat( Title, Host );
  gtk_window_set_title( GTK_WINDOW( window ), Title );
  gtk_signal_connect( GTK_OBJECT (window), "delete_event",
		      (GtkSignalFunc) StatExit, NULL);

  gtk_container_border_width( GTK_CONTAINER( window ), 10 );
#if 0
  vbox = gtk_vbox_new( FALSE, 0 );
  gtk_container_add( GTK_CONTAINER( window ), vbox );
  gtk_widget_show( vbox );

  hbox = gtk_hbox_new( FALSE, 0 );
  gtk_container_add( GTK_CONTAINER( vbox ), hbox );
  gtk_widget_show( hbox );
#endif
  table = gtk_table_new( 7, 2, TRUE );
  gtk_container_add( GTK_CONTAINER( window ), table );

  temp = gtk_label_new( "Offset (us) : " );
  gtk_label_set_justify( GTK_LABEL( temp ),  GTK_JUSTIFY_LEFT );
#if 0
  gtk_box_pack_start( GTK_BOX( hbox ), temp, TRUE, TRUE, 0 );
#endif
  gtk_table_attach_defaults( GTK_TABLE( table ), temp, 0, 1, 0, 1 );
  gtk_widget_show( temp );

  OffsetLabel = gtk_label_new( "--" );
  gtk_label_set_justify( GTK_LABEL( OffsetLabel ),  GTK_JUSTIFY_RIGHT );
  gtk_table_attach_defaults( GTK_TABLE( table ), OffsetLabel, 1, 2, 0, 1 );
  gtk_widget_show( OffsetLabel );

  temp = gtk_label_new( "Microstep (ns) : " );
  gtk_label_set_justify( GTK_LABEL( temp ),  GTK_JUSTIFY_LEFT );
  gtk_table_attach_defaults( GTK_TABLE( table ), temp, 0, 1, 1, 2 );
  gtk_widget_show( temp );

  StepLabel = gtk_label_new( "--" );
  gtk_label_set_justify( GTK_LABEL( StepLabel ),  GTK_JUSTIFY_RIGHT );
  gtk_table_attach_defaults( GTK_TABLE( table ), StepLabel, 1, 2, 1, 2 );
  gtk_widget_show( StepLabel );

  temp = gtk_label_new( "PpsOutput (ns) : " );
  gtk_label_set_justify( GTK_LABEL( temp ),  GTK_JUSTIFY_LEFT );
  gtk_table_attach_defaults( GTK_TABLE( table ), temp, 0, 1, 2, 3 );
  gtk_widget_show( temp );

  PpsLabel = gtk_label_new( "--" );
  gtk_label_set_justify( GTK_LABEL( PpsLabel ),  GTK_JUSTIFY_RIGHT );
  gtk_table_attach_defaults( GTK_TABLE( table ), PpsLabel, 1, 2, 2, 3 );
  gtk_widget_show( PpsLabel );

  temp = gtk_label_new( "Frequency (Hz) : " );
  gtk_label_set_justify( GTK_LABEL( temp ),  GTK_JUSTIFY_LEFT );
  gtk_table_attach_defaults( GTK_TABLE( table ), temp, 0, 1, 3, 4 );
  gtk_widget_show( temp );

  FreqLabel = gtk_label_new( "--" );
  gtk_label_set_justify( GTK_LABEL( FreqLabel ),  GTK_JUSTIFY_RIGHT );
  gtk_table_attach_defaults( GTK_TABLE( table ), FreqLabel, 1, 2, 3, 4 );
  gtk_widget_show( FreqLabel );

  temp = gtk_label_new( "Clock mode : " );
  gtk_label_set_justify( GTK_LABEL( temp ),  GTK_JUSTIFY_LEFT );
  gtk_table_attach_defaults( GTK_TABLE( table ), temp, 0, 1, 4, 5 );
  gtk_widget_show( temp );

  ClkModeLabel = gtk_label_new( "--" );
  gtk_label_set_justify( GTK_LABEL( ClkModeLabel ),  GTK_JUSTIFY_RIGHT );
  gtk_table_attach_defaults( GTK_TABLE( table ), ClkModeLabel, 1, 2, 4, 5 );
  gtk_widget_show( ClkModeLabel );

  button = gtk_button_new_with_label ("Close");
  gtk_signal_connect_object( GTK_OBJECT (button), "clicked",
                             GTK_SIGNAL_FUNC( StatExit ),
                             GTK_OBJECT( window ) );
  gtk_table_attach_defaults( GTK_TABLE( table ), button, 0, 2, 6, 7 );
  GTK_WIDGET_SET_FLAGS( button, GTK_CAN_DEFAULT );
  gtk_widget_grab_default( button );
  gtk_widget_show( button );

  gtk_widget_show( table );
  gtk_widget_show( window );
  StatsWindow = window;
  SendCommand( CLK_GETMICROSTEP );
  SendCommand( CLK_GETPPSOUTPUT );
  SendCommand( CLK_GETHZ );
  SendCommand( CLK_GETCLKMODE );
  return( TRUE );

}


/*
 * Rate window delete event callback.
 */
gint SPH_exit( GtkWidget *widget, GdkEvent *event, gpointer data )
{
  gtk_widget_destroy( widget );
  RateWindow = NULL;
  return( TRUE );
}


void enter_callback( GtkWidget *widget, GtkWidget *entry )
{
  gchar *entry_text;
  int dly;

  entry_text = gtk_entry_get_text( GTK_ENTRY( entry ) );
  if( sscanf( entry_text, "%d", &dly ) == 1 ) {
    Delay = dly;
    if( DelayCounter > Delay )
      DelayCounter = Delay;

  }
  gtk_widget_destroy( RateWindow );
  RateWindow = NULL;

}

/*
 * Sample period menu handler.
 */
gint SamplePeriodHandler( GtkWidget *widget, GdkEventButton *event )
{
  GtkWidget *window;
  GtkWidget *vbox, *hbox;
  GtkWidget *entry, *button, *temp;
  char lBuf[ 256 ];

  if( RateWindow )
    return( TRUE );

  window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
  strcpy( lBuf, "Rate: " );
  strcat( lBuf, Host );
  gtk_window_set_title( GTK_WINDOW( window ), lBuf );
  gtk_signal_connect( GTK_OBJECT (window), "delete_event",
		      (GtkSignalFunc) SPH_exit, NULL);

  gtk_container_border_width( GTK_CONTAINER( window ), 10 );
  vbox = gtk_vbox_new( FALSE, 0 );
  gtk_container_add( GTK_CONTAINER( window ), vbox );
  gtk_widget_show( vbox );

  hbox = gtk_hbox_new( FALSE, 0 );
  gtk_container_add( GTK_CONTAINER( vbox ), hbox );
  gtk_widget_show( hbox );

  temp = gtk_label_new( "Seconds : " );
  gtk_box_pack_start( GTK_BOX( hbox ), temp, TRUE, TRUE, 0 );
  gtk_widget_show( temp );

  entry = gtk_entry_new_with_max_length( 10 );
  gtk_signal_connect( GTK_OBJECT( entry ), "activate",
		      GTK_SIGNAL_FUNC( enter_callback ), entry);
  sprintf( lBuf, "%d", Delay );
  gtk_entry_set_text( GTK_ENTRY( entry ), lBuf );
#if 0
  gtk_entry_select_region( GTK_ENTRY( entry ),
			   0, GTK_ENTRY( entry )->text_length );
#endif
  gtk_box_pack_start( GTK_BOX( hbox ), entry, TRUE, TRUE, 0 );
  gtk_widget_show( entry );

  temp = gtk_label_new( "" );
  gtk_box_pack_start (GTK_BOX (vbox), temp, TRUE, TRUE, 0);
  gtk_widget_show( temp );

  button = gtk_button_new_with_label ("Close");
  gtk_signal_connect_object( GTK_OBJECT (button), "clicked",
                             GTK_SIGNAL_FUNC( SPH_exit ),
                             GTK_OBJECT( window ) );
  gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
  GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
  gtk_widget_grab_default (button);
  gtk_widget_show (button);

  gtk_widget_show(window);
  RateWindow = window;
  return( TRUE );

}


/*
 * Scaling menu handler.
 */
gint SetScaleHandler( GtkWidget *widget )
{
  int i;

  i = (int)(widget);
  if( i == 99 ) {
    AutoScale = 1;
    for( i = 0; ScaleDef[ i ].Value < SampleMax; i++ )
      if( ScaleDef[ i ].Value < 0 )
	break;
  }
  else
    AutoScale = 0;

  if( i != PresentScale ) {
    ScaleMin = -ScaleDef[ i ].Value;
    ScaleMax = ScaleDef[ i ].Value;
    PresentScale = i;
    LinesPerUnit = (double)(Widget->allocation.height - ( 2 * BORDERWIDTH ) ) /
		   (double)(ScaleMax - ScaleMin);
    RedrawSamples();
  }
  return( TRUE );

}


/*
 * Menu handler.
 */
gint DrawMenuHandler( GtkWidget *widget, GdkEventButton *event )
{

  if( event->button == 3 ) {
    GdkEventButton *bevent = (GdkEventButton *) event;

    gtk_menu_popup( GTK_MENU( DrawMenu ), NULL, NULL, NULL, NULL,
		    bevent->button, bevent->time );
    return( TRUE );
  }
  return( FALSE );
}



/*
 * Gdk configure event callback.
 * Create the plot pixmap whenever the size is altered.
 */
gint configure_event( GtkWidget *widget, GdkEventConfigure *event )
{
  GdkColor fg;

  if( Pixmap )
    gdk_pixmap_unref( Pixmap );

  Pixmap = gdk_pixmap_new( widget->window,
			   widget->allocation.width,
			   widget->allocation.height,
			   -1 );
  Widget = widget;
  PixelsPerSample = (double)(widget->allocation.width -
		    ( 7 * BORDERWIDTH ) ) / (double)(N_SAMPLES);
  LinesPerUnit = (double)(widget->allocation.height - ( 2 * BORDERWIDTH ) ) /
		 (double)(ScaleMax - ScaleMin);
  LineWidth = ( widget->allocation.height / 150 + 2 ) | 1;
  if( LineWidth > 5 )
    LineWidth = 5;

#if 0
  printf( "PixelsPerSample:%f LinesPerUnit:%f LineWidth:%d\n",
	  PixelsPerSample, LinesPerUnit, LineWidth );
#endif
  if( !LineGc ) {
    LineGc = gdk_gc_new( widget->window );
    gdk_gc_copy( LineGc, widget->style->black_gc );
    gdk_color_parse( "red", &fg );
    gdk_color_alloc( gtk_widget_get_colormap( widget ), &fg );
    gdk_gc_set_foreground( LineGc, &fg );
  }
  gdk_gc_set_line_attributes( LineGc, LineWidth, GDK_LINE_SOLID,
			      GDK_CAP_ROUND, GDK_JOIN_MITER );
  RedrawSamples();

  return( TRUE );

}

/*
 * Gdk expose event callback for the plot pixmap.
 */
static gint expose_event( GtkWidget *widget, GdkEventExpose *event )
{

  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  Pixmap,
		  event->area.x, event->area.y,
		  event->area.x, event->area.y,
		  event->area.width, event->area.height);

  return( FALSE );

}

/*
 * Gdk delete event callback.
 */
gint delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
    return( FALSE );
}

/*
 * Gdk destroy event callback.
 */
void destroy (GtkWidget *widget, gpointer data)
{
  gtk_main_quit();
}

/*
 * Main entrypoint.
 * Do all setup and then enter the Gdk main loop.
 */
int main( int argc, char *argv[] )
{
  int OptErrors, chr, i;
  char *progname;
  GtkWidget *tWidget;
  GtkWidget *tMenu;
  GtkWidget *window;
  GtkWidget *drawing_area;
  char String[ 256 ];

  /*
   * Check the command line arguments.
   */
  progname = argv[ 0 ];
  OptErrors = 0;
  while( (chr = getopt( argc, argv, "h:st:" )) != EOF) {
    switch( chr ) {

      /*
       * Host name of server to connect to.
       */
      case 'h':
	strncpy( Host, optarg, sizeof Host );
        break;

      /*
       * Ssl mode.
       */
      case 's':
	UseSsl = 1;
        break;

      /*
       * Set time between samples in seconds.
       */
      case 't':
        if( sscanf( optarg, "%d", &Delay ) != 1 ) {
          printf( "Invalid sample time value\n" );
          OptErrors++;
        }
        break;

      default:
        OptErrors++;
        break;
    }
    if( OptErrors ) {
      printf( "usage: %s [-s][-h host][-t time between samples]\n", progname );
      exit(1);
    }
  }
  /*
   * Init ssl if used.
   */
  if( UseSsl ) {
    if( InitSsl( NULL, NULL, NULL, NULL ) != 0 ) {
      printf( "Can'n init ssl\n" );
      exit( 1 );
    }
  }
  /*
   * Do Gdk init.
   */
  gtk_init (&argc, &argv);

  /*
   * Init all windows.
   */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL );
  if( UseSsl ) {
    sprintf( String, "%s (SSL)", Host );
    gtk_window_set_title( GTK_WINDOW( window ), String );
  }
  else
    gtk_window_set_title( GTK_WINDOW( window ), Host );

  gtk_signal_connect( GTK_OBJECT (window), "delete_event",
                      GTK_SIGNAL_FUNC (delete_event), NULL );
  gtk_signal_connect( GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (destroy), NULL );
  gtk_container_border_width( GTK_CONTAINER (window), 0 );
  drawing_area = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA (drawing_area), N_SAMPLES/2, 100);
  gtk_container_add( GTK_CONTAINER (window), drawing_area );
  gtk_widget_show( drawing_area );
  gtk_signal_connect( GTK_OBJECT (drawing_area), "expose_event",
		      (GtkSignalFunc) expose_event, NULL );
  gtk_signal_connect( GTK_OBJECT(drawing_area),"configure_event",
		      (GtkSignalFunc) configure_event, NULL );
  gtk_signal_connect( GTK_OBJECT(drawing_area),"button_press_event",
		      (GtkSignalFunc) DrawMenuHandler, NULL );
  gtk_widget_set_events( drawing_area,
			 GDK_BUTTON_PRESS_MASK |
			 GDK_EXPOSURE_MASK );

  /*
   * Create the popup menu.
   */
  tMenu = gtk_menu_new();
  for( i = 0; DrawMenues[ i ].Selection >= 0; i++ ) {
    tWidget = gtk_menu_item_new_with_label( DrawMenues[ i ].Text );
    gtk_menu_append( GTK_MENU( tMenu ), tWidget );
    gtk_signal_connect_object( GTK_OBJECT( tWidget ), "activate",
			       GTK_SIGNAL_FUNC( DrawMenues[ i ].Handler ),
			       (gpointer)(DrawMenues[ i ].Selection) );
    gtk_widget_show( tWidget );
  }

  DrawMenu = gtk_menu_new();
  tWidget = gtk_menu_item_new_with_label( "Stats " );
  gtk_menu_append( GTK_MENU( DrawMenu ), tWidget );
  gtk_signal_connect_object( GTK_OBJECT( tWidget ), "activate",
			     GTK_SIGNAL_FUNC( StatsHandler ), NULL );
  gtk_widget_show( tWidget );
  tWidget = gtk_menu_item_new_with_label( "Sample Rate " );
  gtk_menu_append( GTK_MENU( DrawMenu ), tWidget );
  gtk_signal_connect_object( GTK_OBJECT( tWidget ), "activate",
			     GTK_SIGNAL_FUNC( SamplePeriodHandler ), NULL );
  gtk_widget_show( tWidget );
  tWidget = gtk_menu_item_new_with_label( "Scaling " );
  gtk_menu_append( GTK_MENU( DrawMenu ), tWidget );
  gtk_menu_item_set_submenu( GTK_MENU_ITEM( tWidget ), tMenu );
  gtk_widget_show( tWidget );
  tWidget = gtk_menu_item_new_with_label( "Quit " );
  gtk_menu_append( GTK_MENU( DrawMenu ), tWidget );
  gtk_signal_connect_object( GTK_OBJECT( tWidget ), "activate",
			     GTK_SIGNAL_FUNC( gtk_main_quit ), NULL );
  gtk_widget_show( tWidget );

  gtk_widget_show( window );

  /*
   * Set termination signal handlers.
   */
  signal( SIGINT, Terminate );
  signal( SIGHUP, Terminate );
  signal( SIGPIPE, SIG_IGN );

  /*
   * Start the server/client protocol.
   * Add Gtk i/o and timer handlers .
   */
  if( ( ReadFd = MicrodConnect( Host ) ) < 0 ) {
    printf( "Connect error\n" );
    exit( 1 );
  }
  if( UseSsl ) {
    Ssl = SSL_new( SslCtx );
    if( Ssl == NULL ) {
      printf( "Can't allocate ssl struct\n" );
      close( ReadFd );
      exit( 1 );
    }
    SSL_set_fd( Ssl, ReadFd );
    SSL_connect( Ssl );
  }
  DelayCounter = Delay;
  gtk_timeout_add( 1000, (GtkFunction)(TickTock), NULL );
  gdk_input_add( ReadFd, GDK_INPUT_READ,
		 (GdkInputFunction)(ReadMicrod), NULL );
  SendCommand( CLK_GETMED );

  /*
   * Enter Gtk main loop.
   */
  gtk_main();
  if( UseSsl ) {
    SSL_shutdown( Ssl );
    shutdown( SSL_get_fd( Ssl ), 2 );
    close( ReadFd );
    if( Ssl != NULL )
      SSL_free( Ssl );

    if( SslCtx != NULL )
      SSL_CTX_free( SslCtx );
  }
  else
    close( ReadFd );

  return 0 ;
}
