#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <time.h>
#include <string.h>
#include <syslog.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include "ppsclock.h"

int PpsFd;


/*
 * Terminate the program.
 */
static void Terminate( int sig )
{

  signal(SIGTERM, SIG_DFL);
  close( PpsFd );
  exit( 0 );

}


/*
 * Disconnect the tty and enter daemon mode.
 */
static void DisconnectFromTty()
{
  int i;

  if( fork() ) {
    exit(0);
  }
  i = open("/dev/tty", O_RDWR);
  if( i >= 0 ) {
    (void) ioctl(i, TIOCNOTTY, NULL);
    (void) close(i);
  }

}

/*
 * Display usage syntax.
 */
static void Usage( char *Prog )
{

  printf( "Usage: %s [-d][-t][-r][-s interval in seconds] [-f file]\n", Prog );
  exit( 1 );

}

int main( int argc, char *argv[] )
{
  int OptErrors, chr, Median, Today;
  int LogToFile = 0, TimeStamp = 0, DateStamp = 0, Rotate = 0;
  time_t Interval = 60L;
  time_t Now;
  struct tm *tPnt;
  FILE *LogFp;
  char *progname;
  char LogFile[ 256 ];
  char FileTemp[ 256 ];

  /*
   * Check the command line arguments.
   */
  progname = argv[ 0 ];
  OptErrors = 0;
  while( (chr = getopt( argc, argv, "df:rs:t" )) != EOF) {
    switch( chr ) {

      /*
       * Cleartext date on log records.
       */
      case 'd':
	DateStamp = 1;
        break;

      /*
       * Name of log file.
       */
      case 'f':
	strncpy( LogFile, optarg, sizeof LogFile );
	LogToFile = 1;
        break;

      /*
       * If logging to file, append date to file name
       * and create a new file each day.
       */
      case 'r':
	Rotate = 1;
        break;

      /*
       * Seconds between log samples.
       */
      case 's':
        if( sscanf( optarg, "%ld", &Interval ) != 1 ) {
          printf( "Invalid interval time value\n" );
          OptErrors++;
        }
        break;

      /*
       * Include timestamps in log file.
       */
      case 't':
	TimeStamp = 1;
        break;

      default:
        OptErrors++;
        break;
    }
  }
  if( OptErrors ) {
    Usage( progname );
  }
  if( ( PpsFd = open( "/dev/ppsd0", O_RDWR ) ) < 0 ) {
    printf( "%s: Can't open /dev/ppsd0\n", progname );
    exit( 1 );
  }
  (void)signal(SIGTERM, Terminate);
  (void)signal(SIGINT, Terminate);
  (void)signal(SIGQUIT, Terminate);
  /*
   * Open syslog and disconnect from the controlling tty.
   */
  if( LogToFile ) {
    if( Rotate ) {
      Now = time( NULL );
      tPnt = localtime( &Now );
      snprintf( FileTemp, sizeof FileTemp, "%s%04d%02d%02d", LogFile,
		  tPnt->tm_year + 1900, tPnt->tm_mon + 1, tPnt->tm_mday );
      if( ( LogFp = fopen( FileTemp, "w" ) ) == NULL ) {
        printf( "%s: Can't create log file %s\n", progname, FileTemp );
        exit( 1 );
      }
      Today = tPnt->tm_mday;
    }
    else if( ( LogFp = fopen( LogFile, "w" ) ) == NULL ) {
      printf( "%s: Can't create log file %s\n", progname, LogFile );
      exit( 1 );
    }
    openlog( "logd", LOG_PID | LOG_NDELAY, LOG_LOCAL0 );
    syslog( LOG_ERR, "Starting log to file %s", LogFile );
    DisconnectFromTty();
  }
  for( ; ; ) {
    Now = time( NULL );
    tPnt = localtime( &Now );
    if( Rotate && Today != tPnt->tm_mday ) {
      fclose( LogFp );
      snprintf( FileTemp, sizeof FileTemp, "%s%04d%02d%02d", LogFile,
		  tPnt->tm_year + 1900, tPnt->tm_mon + 1, tPnt->tm_mday );
      if( ( LogFp = fopen( FileTemp, "w" ) ) == NULL ) {
	if( LogToFile )
	  syslog( LOG_ERR, "Can't create log file %s", FileTemp );
	else
	  printf( "%s: Can't create log file %s\n", progname, FileTemp );

	exit( 1 );
      }
      Today = tPnt->tm_mday;
    }
    if( ioctl(PpsFd, CIOGETMED, (caddr_t)&Median) < 0 ) {
      if( LogToFile )
	syslog( LOG_ERR, "Can't read value from pps driver" );
      else
	printf( "%s: Can't read value from pps driver\n", progname );
      close( PpsFd );
      exit( 1 );
    }
    if( TimeStamp ) {
      if( LogToFile ) {
	fprintf( LogFp, "%ld %d\n", Now, Median );
	fflush( LogFp );
      }
      else
	printf( "%ld %d\n", Now, Median );
    }
    else if( DateStamp ) {
      if( LogToFile ) {
	fprintf( LogFp, "%04d-%02d-%02d %02d:%02d:%02d %.3f\n",
		 tPnt->tm_year + 1900, tPnt->tm_mon + 1, tPnt->tm_mday,
		 tPnt->tm_hour, tPnt->tm_min, tPnt->tm_sec,
                 (double)(Median) / 1000.0 );
	fflush( LogFp );
      }
      else
	printf( "%04d-%02d-%02d %02d:%02d:%02d %.3f\n",
		tPnt->tm_year + 1900, tPnt->tm_mon + 1, tPnt->tm_mday,
		tPnt->tm_hour, tPnt->tm_min, tPnt->tm_sec,
                (double)(Median) / 1000.0 );
    }
    else {
      if( LogToFile ) {
	fprintf( LogFp, "%.3f\n", (double)(Median) / 1000.0 );
	fflush( LogFp );
      }
      else
	printf( "%.3f\n", (double)(Median) / 1000.0 );
    }
    while( time( NULL ) < Now + Interval )
      usleep( 100000 );
  }
  return 0;
}
