/*
 * Crude hack to download HW code to Atmel eeproms using a i2c bus
 * connected to the PC printer port.
 *
 * The cable used to connect the printer port to the 10-pole
 * connector on the card is connected as follows :
 *
 *	10 pin Fpga connector		DB25 PC printer port
 *	---------------------		--------------------
 *		Data   1 -------------------- 1
 *		Clock  3 -------------------- 14
 *		GND    7 -------------------- 25
 *              SE    10 -------------------- 17
 *
 * The Data, Clock and SE signals are available at BASEADDRESS + 2
 * which usuallay is 0x37a, 0x27a or 0x3be
 *
 *	Bit 0:	Inverted Data.
 *	Bit 1:	Inverted Clock.
 *	Bit 2:	Unused.
 *	Bit 3:	Inverted SE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/ttycom.h>
#include <machine/console.h>
#include <machine/cpufunc.h>


/*
 * Default for port address.
 */
#ifndef PORT
#define PORT			0x378
#endif


/*
 * Printer port pins used.
 */
#define ALTERA_DATA		0x01
#define ALTERA_CLOCK		0x02
#define ALTERA_SE		0x08


/*
 * Eeprom page size.
 */
#define ALTERA_PAGESIZE		128


/*
 * Constants for read and write mode.
 */
#define ALTERA_WRITEMODE	0
#define ALTERA_READMODE		1


/*
 * Constants for lsb and msb mode.
 */
#define ALTERA_MSBMODE		0
#define ALTERA_LSBMODE		1


/*
 * Internal config addresses.
 */
#define ALTERA_RESETPOLADDR	0x020000
#define ALTERA_MFGIDADDR	0x040000
#define ALTERA_CLOCKADDR	0x380000


/*
 * Max number of error printouts.
 */
#define ALTERA_MAXERRORS	64


/*
 * Global static data.
 */
int PrintPort = PORT;
unsigned char PortData = 0;
struct timeval ProgStart;
int DataLength = 0;
unsigned char DataBuffer[ 128 * 1024 ];
unsigned char VerifyBuffer[ 128 * 1024 ];


/*
 * Enable IOPL.
 *
 * Return 0 if OK, 1 for errors.
 */
static int EnableIopl()
{
  int fd;

  if( ( fd = open( "/dev/ttyv0", O_RDWR ) ) < 0 ) {
    printf( "Can't open /dev/ttyv0\n" );
    return( 1 );
  }
  if( ioctl( fd, KDENABIO, 0 ) < 0 ) {
    printf( "Can't acquire iopl\n" );
    return( 1 );
  }
  close( fd );
  return( 0 );
}


/*
 * Disable IOPL.
 *
 * Return 0 if OK, 1 for errors.
 */
static int DisableIopl()
{
  int fd;

  if( ( fd = open( "/dev/ttyv0", O_RDWR ) ) < 0 ) {
    printf( "Can't open /dev/ttyv0\n" );
    return( 1 );
  }
  if( ioctl( fd, KDDISABIO, 0 ) < 0 ) {
    printf( "Can't disable iopl\n" );
    return( 1 );
  }
  close( fd );
  return( 0 );
}


/*
 * Terminate the program.
 */
static void Terminate( int sig )
{

  signal( SIGTERM, SIG_DFL );
  signal( SIGINT, SIG_DFL );
  signal( SIGQUIT, SIG_DFL );
  outb( PrintPort, 0 );
  DisableIopl();
  printf( "\nTerminating\n" );
  exit( 1 );

}


/*
 * Send a start condition.
 */
static void SendStartCondition()
{

  outb( PrintPort, ALTERA_SE );
  inb( PrintPort );
  inb( PrintPort );
  outb( PrintPort, ALTERA_SE | ALTERA_DATA );
  inb( PrintPort );
  inb( PrintPort );
  outb( PrintPort, ALTERA_SE | ALTERA_DATA | ALTERA_CLOCK );
  inb( PrintPort );
  inb( PrintPort );
  PortData = ALTERA_SE | ALTERA_DATA | ALTERA_CLOCK;

}


/*
 * Send a stop condition.
 */
static void SendStopCondition()
{

  outb( PrintPort, ALTERA_SE | ALTERA_DATA );
  inb( PrintPort );
  inb( PrintPort );
  outb( PrintPort, ALTERA_SE );
  inb( PrintPort );
  inb( PrintPort );
  outb( PrintPort, ALTERA_SE | ALTERA_CLOCK );
  inb( PrintPort );
  inb( PrintPort );
  PortData = ALTERA_SE | ALTERA_CLOCK;

}

/*
 * Send a bit.
 */
static void SendBit( int Bit )
{

  if( ( PortData & ALTERA_CLOCK ) == 0 ) {
    PortData |= ALTERA_CLOCK;
    outb( PrintPort, PortData );
    inb( PrintPort );
    inb( PrintPort );
  }
  if( Bit )
    PortData &= ~ALTERA_DATA;
  else
    PortData |= ALTERA_DATA;

  outb( PrintPort, PortData );
  inb( PrintPort );
  PortData &= ~ALTERA_CLOCK;
  outb( PrintPort, PortData );
  inb( PrintPort );
  PortData |= ALTERA_CLOCK;
  outb( PrintPort, PortData );
  inb( PrintPort );

}

/*
 * Read a bit.
 *
 * Return 0 or 1 according to the read bit.
 */
static int ReadBit()
{
  int RetData;

  if( ( PortData & ALTERA_CLOCK ) == 0 )
    PortData |= ALTERA_CLOCK;

  if( ( PortData & ALTERA_DATA ) != 0 )
    PortData &= ~ALTERA_DATA;

  outb( PrintPort, PortData );
  inb( PrintPort );
  PortData &= ~ALTERA_CLOCK;
  outb( PrintPort, PortData );
  inb( PrintPort );

  if( inb( PrintPort ) & ALTERA_DATA )
    RetData = 0;
  else
    RetData = 1;

  PortData |= ALTERA_CLOCK;
  outb( PrintPort, PortData );
  inb( PrintPort );
  return( RetData );

}


/*
 * Read a byte.
 */
static unsigned char ReadByte()
{
  unsigned char Chr = 0;
  int i;

  for( i = 0; i < 8; i++ ) {
    Chr = Chr >> 1;
    if( ReadBit() != 0 )
      Chr |= 0x80;

  }
  return( Chr );

}


/*
 * Send 8 bits.
 *
 * Return the received Ack bit.
 * The Ack bit is 0 if OK, 1 for errors.
 */
static int SendByte( unsigned char Byte, int LsbFirst )
{
  int i;
  unsigned char Mask;

  if( LsbFirst == ALTERA_LSBMODE ) {
    Mask = 0x01;
    for( i = 0; i < 8; i++ ) {
      if( Byte & Mask )
	SendBit( 1 );
      else
	SendBit( 0 );

      Mask = Mask << 1;
    }
  }
  else {
    Mask = 0x80;
    for( i = 0; i < 8; i++ ) {
      if( Byte & Mask )
	SendBit( 1 );
      else
	SendBit( 0 );

      Mask = Mask >> 1;
    }
  }
  return( ReadBit() );

}


/*
 * Send the device address.
 *
 * Return 0 if OK, 1 for errors.
 */
static int SendDeviceAddress( int Read )
{
  unsigned char DevAddr = 0xae;

  if( Read == ALTERA_READMODE )
    DevAddr |= 0x01;

  return( SendByte( DevAddr, 0 ) );

}


/*
 * Send the eeprom address.
 *
 * Return 0 if OK, 1 for errors.
 */
static int SendPromAddress( int Address )
{

  if( SendByte((unsigned char)((Address >> 16) & 0xff), ALTERA_MSBMODE) != 0 )
    return( 1 );

  if( SendByte( (unsigned char)((Address >> 8) & 0xff), ALTERA_MSBMODE ) != 0 )
    return( 1 );

  if( SendByte( (unsigned char)(Address & 0xff), ALTERA_MSBMODE ) != 0 )
    return( 1 );

  return( 0 );
}


/*
 * Wait for the eeprom to get ready.
 *
 * Return 0 if OK, 1 for timeout. ( 200 ms ).
 */
static int WaitUntilReady( int Read )
{
  struct timeval tmvl;
  long Now, Max;

  gettimeofday( &tmvl, NULL );
  Max = ( tmvl.tv_sec - ProgStart.tv_sec ) * 1000000L + tmvl.tv_usec;
  Max += 200000L;
  Now = Max;
  do {
    SendStartCondition();
    if( SendDeviceAddress( Read ) == 0 )
      return( 0 );

    gettimeofday( &tmvl, NULL );
    Now = ( tmvl.tv_sec - ProgStart.tv_sec ) * 1000000L + tmvl.tv_usec;
  }
  while( Now < Max );
  return( 1 );

}


/*
 * Send a page of data.
 *
 * Return 0 if OK, 1 for errors.
 */
static int SendDataPage( int Address, unsigned char *Page )
{
  int Count;

  if( WaitUntilReady( ALTERA_WRITEMODE ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  if( SendPromAddress( Address ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  for( Count = 0; Count < ALTERA_PAGESIZE; Count++ ) {
    if( SendByte( *Page++, ALTERA_LSBMODE ) != 0 ) {
      SendStopCondition();
      return( 1 );
    }
  }
  SendStopCondition();
  return( 0 );

}


/*
 * Read data.
 *
 * Return 0 if OK, 1 for errors.
 */
static int ReadData( int Address, unsigned char *Page, int Len )
{
  int Count;

  if( WaitUntilReady( ALTERA_WRITEMODE ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  if( SendPromAddress( Address ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  if( WaitUntilReady( ALTERA_READMODE ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  for( Count = 0; Count < Len; Count++ ) {
    *Page++ = ReadByte();
    SendBit( 0 );
  }
  SendStopCondition();
  return( 0 );

}


/*
 * Write the entire file to the eeprom.
 */
static int WriteFpga()
{
  unsigned char *BufPnt;
  int i;

  BufPnt = DataBuffer;
  for( i = 0; i < DataLength; i += ALTERA_PAGESIZE ) {
    if( SendDataPage( i, BufPnt ) != 0 ) {
      printf( "Error sending data page\n" );
      return( 1 );
    }
    BufPnt += ALTERA_PAGESIZE;
  }
  if( WaitUntilReady( ALTERA_READMODE ) != 0 ) {
    printf( "Error polling write\n" );
    return( 1 );
  }
  return( 0 );

}


/*
 * Verify the entire file against the eeprom.
 */
static int VerifyFpga()
{
  int i, Error;

  Error = 0;
  if( ReadData( 0, VerifyBuffer, DataLength ) != 0 ) {
    printf( "Error reading data\n" );
    return( 1 );
  }
  for( i = 0; i < DataLength; i++ ) {
    if( DataBuffer[ i ] != VerifyBuffer[ i ] ) {
      Error++;
      printf( "Error in byte %d should be %02x but is %02x\n",
	      i, DataBuffer[ i ], VerifyBuffer[ i ] );
      if( Error > ALTERA_MAXERRORS )
	return( 1 );
    }
  }
  if( Error != 0 )
    return( 1 );
  else
    return( 0 );

}


/*
 * Write clock information to the eeprom.
 */
static int WriteClockMode( int On )
{

  unsigned char OnOff;

  if( On )
    OnOff = 0xff;
  else
    OnOff = 0x00;

  SendStartCondition();
  if( WaitUntilReady( ALTERA_WRITEMODE ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  if( SendPromAddress( ALTERA_CLOCKADDR ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  if( SendByte( OnOff, ALTERA_LSBMODE ) != 0 ) {
      SendStopCondition();
      return( 1 );
    }
  SendStopCondition();
  if( WaitUntilReady( ALTERA_READMODE ) != 0 ) {
    printf( "Error polling WriteClock\n" );
    return( 1 );
  }
  return( 0 );

}


/*
 * Write the reset polarity to the eeprom.
 */
static int WriteResetPolarity( int Polarity )
{
  int Count;
  unsigned char PolData;

  if( Polarity )
    PolData = 0x00;
  else
    PolData = 0xff;

  SendStartCondition();
  if( WaitUntilReady( ALTERA_WRITEMODE ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  if( SendPromAddress( ALTERA_RESETPOLADDR ) != 0 ) {
    SendStopCondition();
    return( 1 );
  }
  for( Count = 0; Count < 4; Count++ ) {
    if( SendByte( PolData, ALTERA_LSBMODE ) != 0 ) {
      SendStopCondition();
      return( 1 );
    }
  }
  SendStopCondition();
  if( WaitUntilReady( ALTERA_READMODE ) != 0 ) {
    printf( "Error polling write\n" );
    return( 1 );
  }
  return( 0 );

}


/*
 * Read id bytes.
 */
static int ReadId()
{
  int i;

  if( ReadData( ALTERA_MFGIDADDR, VerifyBuffer, 2 ) != 0 ) {
    printf( "Error reading id\n" );
    return( 1 );
  }
  printf( "Id : " );
  for( i = 0; i < 2; i++ )
    printf( "%02x ", VerifyBuffer[ i ] );

  printf( "\n" );
  return( 0 );

}


/*
 * Read id and reset polarity bytes.
 */
static int ReadResetPolarity()
{
  int i;

  if( ReadData( ALTERA_RESETPOLADDR, VerifyBuffer, 4 ) != 0 ) {
    printf( "Error reading id\n" );
    return( 1 );
  }
  printf( "Reset polarity : " );
  for( i = 0; i < 4; i++ )
    printf( "%02x ", VerifyBuffer[ i ] );

  printf( "\n" );
  return( 0 );

}


/*
 * Decode 2 hex characters and store them in a binary byte.
 */
static int DecodeHex( char *AsciiBuf, unsigned char *BinaryBuf, int Shift )
{
  unsigned char BinByte, Chr;

  if( !isxdigit( AsciiBuf[ 0 ] ) || !isxdigit( AsciiBuf[ 1 ] ) )
    return( 1 );

  Chr = toupper( AsciiBuf[ 0 ] );
  if( Chr > '9' )
    BinByte = ( Chr - 7 ) << 4;
  else
    BinByte = Chr << 4;

  Chr = toupper( AsciiBuf[ 1 ] );
  if( Chr > '9' )
    BinByte |= ( ( Chr - 7 ) & 0x0f );
  else
    BinByte |= ( Chr & 0x0f );

  if( Shift ) {
    *BinaryBuf = ( BinByte >> 1 ) | ( *BinaryBuf & 0x80 );
    BinaryBuf++;
    *BinaryBuf = BinByte << 7;
  }
  else {
    *BinaryBuf = BinByte;
  }
  return( 0 );

}


/*
 * Read a Intelhex file.
 */
static int ReadFile( char *FileName, int Shift )
{
  int Offset = 0;
  int MaxAddress = 0;
  int LoadOffset;
  int Address, RecLen, i;
  char Buffer[ 256 ];
  FILE *fp;

  if( ( fp = fopen( FileName, "r" ) ) == NULL ) {
    printf( "Can't open file %s\n", FileName );
    return( 1 );
  }

  /*
   * Read hex file line by line.
   */
  bzero( Buffer, 16 );
  while( fgets( Buffer, sizeof Buffer, fp ) != NULL ) {

    /*
     * All lines must start with a ':' character.
     */
    if( Buffer[ 0 ] != ':' ) {
      printf( "No \":\" record mark found\n" );
      fclose( fp );
      return( 1 );
    }

    /*
     * All record type values must start with '0'.
     */
    if( Buffer[ 7 ] != '0' ) {
      printf( "Unknown record type\n" );
      fclose( fp );
      return( 1 );
    }

    /*
     * Offset record ?
     */
    if( Buffer[ 8 ] == '4' ) {
      Buffer[ 13 ] = '\0';
      if( sscanf( &Buffer[ 9 ], "%x", &Offset ) != 1 ) {
	printf( "Invalid offset record\n" );
	fclose( fp );
	return( 1 );
      }
      Offset = Offset << 16;
      printf( "File offset %d\n", Offset );
    }

    /*
     * End of file record ?
     */
    else if( Buffer[ 8 ] == '1' ) {
      printf( "End of file record found\n" );
      break;
    }

    /*
     * Data record ?
     */
    else if( Buffer[ 8 ] == '0' ) {
      Buffer[ 7 ] = '\0';
      if( sscanf( &Buffer[ 3 ], "%x", &LoadOffset ) != 1 ) {
	printf( "Invalid offset in data record\n" );
	fclose( fp );
	return( 1 );
      }
      Buffer[ 3 ] =  '\0';
      if( sscanf( &Buffer[ 1 ], "%x", &RecLen ) != 1 ) {
	printf( "Invalid reclen in data record\n" );
	fclose( fp );
	return( 1 );
      }
      Address = Offset + LoadOffset;

      /*
       * Keep track of max load address.
       */
      if( Address + RecLen > MaxAddress )
	MaxAddress = Address + RecLen;

      for( i = 0; i < RecLen; i++ ) {
	if( DecodeHex( &Buffer[ i * 2 + 9 ],
		       &DataBuffer[ Address + i ], Shift ) != 0 ) {
	  printf( "Invalid hex data in data record\n" );
	  fclose( fp );
	  return( 1 );
	}
      }
    }

    /*
     * All other record types are invalid.
     */
    else {
      printf( "Unknown record type\n" );
      fclose( fp );
      return( 1 );
    }
    bzero( Buffer, 16 );
  }
  fclose( fp );

  /*
   * Shifted data add one to max load address.
   */
  if( Shift )
    DataLength = MaxAddress + 1;
  else
    DataLength = MaxAddress;

  printf( "DataLength %d\n", DataLength );
#if 0
  fp = fopen( "test.hex", "w" );
  for( i = 0; i < DataLength; i++ ) {
    if( ( i % 43 ) == 0 )
      fprintf( fp, "\n" );
    fprintf( fp, "%02x ", DataBuffer[ i ] );
  }
  fprintf( fp, "\n" );
  fclose( fp );
#endif
  return( 0 );

}


/*
 * Main entrypoint, process command line args, do the beacon
 * handshaking, get the directory listing and finally read
 * all files from the camera.
 */
int main( int argc, char *argv[] )
{
  int OptErrors, chr, SetResetPolarity, Verify, IdRead, Shift;
  int SetClockMode, ResetPolarity, ClockMode;
  char *progname, FileName[ 256 ] = "";

  /*
   * Check the command line arguments.
   */
  SetResetPolarity = Verify = IdRead = Shift= 0;
  SetClockMode = ResetPolarity = ClockMode = 0;
  progname = argv[ 0 ];
  OptErrors = 0;
  while( ( chr = getopt( argc, argv, "c:f:ip:r:sv" ) ) != EOF ) {
    switch( chr ) {

      /*
       * Set clock mode.
       */
      case 'c':
	if( sscanf( optarg, "%d", &ClockMode ) != 1
	|| ( ClockMode != 0 && ClockMode != 1 ) ) {
	  printf( "Invalid clock mode (0 1)\n" );
	  OptErrors++;
	}
	SetClockMode = 1;
        break;

      /*
       * Set printer port address.
       */
      case 'f':
	strncpy( FileName, optarg, sizeof FileName );
        break;

      /*
       * Read mfg/device id and reset polarity.
       */
      case 'i':
	IdRead = 1;
        break;

      /*
       * Set printer port base address.
       */
      case 'p':
	if( sscanf( optarg, "%x", &PrintPort ) != 1
	|| (PrintPort != 0x378 && PrintPort != 0x278 && PrintPort != 0x3bc) ) {
	  printf( "Invalid port address (278 378 3bc)\n" );
	  OptErrors++;
	}
        break;

      /*
       * Set reset polarity.
       */
      case 'r':
	if( sscanf( optarg, "%d", &ResetPolarity ) != 1
	|| ( ResetPolarity != 0 && ResetPolarity != 1 ) ) {
	  printf( "Invalid reset polarity (0 1)\n" );
	  OptErrors++;
	}
	SetResetPolarity = 1;
        break;

      /*
       * Shift bit pattern when reading hex file.
       */
      case 's':
	Shift = 1;
        break;

      /*
       * Verify eeprom against hex file.
       */
      case 'v':
	Verify = 1;
        break;

      default:
        OptErrors++;
        break;
    }
    if( OptErrors ) {
      printf( "usage: %s [-p port address][-r 0/1][-c 0/1][-i][-v][-s] -f filename\n",
	      progname );
      exit(1);
    }
  }
  /*
   * Read the file to memory.
   */
  if( !SetResetPolarity && !SetClockMode && !IdRead ) {
    if( ReadFile( FileName, Shift ) != 0 )
      exit( 1 );
  }

  /*
   * Redirect some signals.
   */
  (void)signal( SIGHUP, SIG_IGN );
  (void)signal( SIGTERM, Terminate );
  (void)signal( SIGINT, Terminate );
  (void)signal( SIGQUIT, Terminate );

  /*
   * Read timeofday to be used for time calculations and enable I/O.
   */
  gettimeofday( &ProgStart, NULL );
  if( EnableIopl() != 0 )
    exit( 2 );

  /*
   * Add offset to printer port address and set clock/data low.
   */
  PrintPort += 2;
  outb( PrintPort, ALTERA_SE | ALTERA_CLOCK );
  PortData = ALTERA_SE | ALTERA_CLOCK;

  /*
   * Perform task according to command line arguments.
   */
  if( IdRead ) {
      printf( "Reading Manufacturers device codes\n" );
      if( ReadId() != 0 )
	printf( "Error reading id\n" );

      printf( "Reading reset polarity\n" );
      if( ReadResetPolarity() != 0 )
	printf( "Error reset polarity\n" );
  }
  else if( SetResetPolarity ) {
    printf( "Setting reset polarity = %d\n", ResetPolarity );
    if( WriteResetPolarity( ResetPolarity ) != 0 )
      printf( "Error writing reset polarity\n" );
  }
  else if( SetClockMode ) {
    printf( "Setting clock mode = %d\n", ClockMode );
    if( WriteClockMode( ClockMode ) != 0 )
      printf( "Error writing clock mode\n" );
  }
  else if( Verify ) {
    printf( "Verifying eeprom data\n" );
    if( VerifyFpga() != 0 )
      printf( "Error verifying eeprom data\n" );
  }
  else {
    printf( "Writing eeprom data\n" );
    if( WriteFpga() != 0 )
      printf( "Error writing eeprom data\n" );
  }

  /*
   * Set clock/data high, keep I/O enabled and exit.
   */
  outb( PrintPort, ALTERA_SE );
  DisableIopl();
  return 0;

}
