/*
 * Test of serial programming of a ICD2061A
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/ttycom.h>
#include <machine/console.h>
#include <machine/cpufunc.h>


/*
 * Default for port address.
 */
#ifndef PORT
#define PORT 0x3e6
#endif


/*
 * Serial port definitions.
 */
#define ICD_CLOCK	0x01
#define ICD_DATA	0x02


#define SPEED_CONST	0x64f094


/*
 * Enable IOPL.
 *
 * Return 0 if OK, 1 for errors.
 */
static int EnableIopl()
{
  int fd;

  if( ( fd = open( "/dev/ttyv0", O_RDWR ) ) < 0 ) {
    printf( "Can't open /dev/ttyv0\n" );
    return( 1 );
  }
  if( ioctl( fd, KDENABIO, 0 ) < 0 ) {
    printf( "Can't acquire iopl\n" );
    return( 1 );
  }
  close( fd );
  return( 0 );
}


/*
 * Disable IOPL.
 *
 * Return 0 if OK, 1 for errors.
 */
static int DisableIopl()
{
  int fd;

  if( ( fd = open( "/dev/ttyv0", O_RDWR ) ) < 0 ) {
    printf( "Can't open /dev/ttyv0\n" );
    return( 1 );
  }
  if( ioctl( fd, KDDISABIO, 0 ) < 0 ) {
    printf( "Can't disable iopl\n" );
    return( 1 );
  }
  close( fd );
  return( 0 );
}


/*
 * Send a unlock command.
 */
static void Unlock()
{
  int i;

  /*
   * Unlock sequence.
   */
  outb( PORT, ICD_CLOCK | ICD_DATA );
  inb( PORT );
  inb( PORT );
  for( i = 0; i < 5; i++ ) {
    outb( PORT, ICD_DATA );
    inb( PORT );
    outb( PORT, ICD_CLOCK | ICD_DATA );
    inb( PORT );
  }
  outb( PORT, ICD_DATA );
  inb( PORT );
  outb( PORT, 0 );
  inb( PORT );
  outb( PORT, ICD_CLOCK );
  inb( PORT );

}


/*
 * Send a stop bit.
 */
static void Stopbit()
{

  outb( PORT, ICD_CLOCK | ICD_DATA );
  inb( PORT );
  outb( PORT, ICD_DATA );
  inb( PORT );
  outb( PORT, ICD_CLOCK | ICD_DATA );
  inb( PORT );
  inb( PORT );
  inb( PORT );
  inb( PORT );

}


/*
 * Send a start bit.
 */
static void Startbit()
{

  outb( PORT, 0 );
  inb( PORT );
  outb( PORT, ICD_CLOCK );
  inb( PORT );

}


/*
 * Send a single bit.
 */
static void Sendbit( int Bit )
{

  if( Bit )
    outb( PORT, ICD_CLOCK );
  else
    outb( PORT, ICD_CLOCK | ICD_DATA );

  inb( PORT );
  if( Bit )
    outb( PORT, 0 );
  else
    outb( PORT, ICD_DATA );

  inb( PORT );
  if( Bit )
    outb( PORT, ICD_DATA );
  else
    outb( PORT, 0 );

  inb( PORT );
  if( Bit )
    outb( PORT, ICD_CLOCK | ICD_DATA );
  else
    outb( PORT, ICD_CLOCK );

  inb( PORT );
  

}


/*
 * Transfer the command to the icd2061.
 */
static void ProgramFreq( int Value )
{
  int i;

  Unlock();
  Startbit();
  for( i = 0; i < 24; i++ ) {
    if( Value & 0x01 )
      Sendbit( 1 );
    else
      Sendbit( 0 );

    Value = Value >> 1;
  }
  Stopbit();
  outb( PORT, 0 );
}


/*
 * Main entrypoint, process command line args, do the beacon
 * handshaking, get the directory listing and finally read
 * all files from the camera.
 */
int main( int argc, char *argv[] )
{

  if( EnableIopl() != 0 )
    exit( 2 );

  ProgramFreq( SPEED_CONST );

  DisableIopl();
  return 0;

}
