
/*
 * refclock_ppsdevice - clock driver for 1-pps signals
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <ctype.h>

#include "ntpd.h"
#include "ntp_io.h"
#include "ntp_unixtime.h"
#include "ntp_refclock.h"
#include "ntp_stdlib.h"

#include "ppsclock.h"

#if defined(REFCLOCK) && defined(CLOCK_PPSDEVICE)

# ifdef HAVE_TIMEPPS_H
#  include <timepps.h>
# else
#  ifdef HAVE_SYS_TIMEPPS_H
#   include <sys/timepps.h>
#  endif
# endif

/*
 * This driver supports a specialized PPS interface card developed
 * for the swedish internet time distribution project.
 *
 * The driver is a adapted and simplified version of the refclock_atom driver.
 *
 * This version distributes nanosecond timestamps from the kernel driver
 * instead of the microsecond timestamps of the previous version.
 *
 * Fudge Factors
 *
 * There are no special fudge factors other than the generic.
 *
 */
/*
 * Interface definitions
 */
#define DEVICE		"/dev/ppsd%d" /* device name and unit */

#define	PRECISION	(-27)	/* precision assumed (about 1 us) */
#define	REFID		"PPS\0"	/* reference ID */
#define	DESCRIPTION	"PPS Device driver" /* WRU */
#define NANOSECOND	1000000000 /* one second (ns) */
#define RANGEGATE	500000	/* range gate (ns) */
#define ASTAGE		8	/* filter stages */

static struct peer *pps_peer;	/* ppsdevice driver for PPS sources */

/*
 * PPS unit control structure
 */
struct ppsunit {
	struct timespec ts;	/* last timestamp */
	int fddev;		/* pps device descriptor */
	pps_params_t pps_params; /* pps parameters */
	struct ppsclockev ev;
	pps_handle_t handle;	/* pps handlebars */
};

/*
 * Function prototypes
 */
static	int	ppsdevice_start	P((int, struct peer *));
static	void	ppsdevice_poll	P((int, struct peer *));
static	void	ppsdevice_shutdown	P((int, struct peer *));
static	int	ppsdevice_pps	P((struct peer *));

/*
 * Transfer vector
 */
struct	refclock refclock_ppsdevice = {
	ppsdevice_start,		/* start up driver */
	ppsdevice_shutdown,		/* shut down driver */
	ppsdevice_poll,		/* transmit poll message */
	noentry,		/* fudge control */
	noentry,		/* initialize driver */
	noentry,		/* not used (old ppsdevice_buginfo) */
	NOFLAGS			/* not used */
};


/*
 * ppsdevice_start - initialize data for processing
 */
static int
ppsdevice_start(
	int unit,		/* unit number (not used) */
	struct peer *peer	/* peer structure pointer */
	)
{
	struct refclockproc *pp;
	register struct ppsunit *up;
	char device[80];

	/*
	 * Allocate and initialize unit structure
	 */
	pps_peer = peer;
	pp = peer->procptr;
	peer->precision = PRECISION;
	pp->clockdesc = DESCRIPTION;
	pp->stratum = STRATUM_REFCLOCK;
	memcpy((char *)&pp->refid, REFID, 4);
	peer->burst = ASTAGE;
	up = emalloc(sizeof(struct ppsunit));
	memset(up, 0, sizeof(struct ppsunit));
	pp->unitptr = (caddr_t)up;

	/*
	 * Open PPS device.
	 */
	sprintf(device, DEVICE, unit);
	up->fddev = open(device, O_RDWR, 0777);
	if (up->fddev <= 0) {
		msyslog(LOG_ERR,
		    "refclock_ppsdevice: %s: %m", device);
		return (0);
	}
	return (1);
}


/*
 * ppsdevice_shutdown - shut down the clock
 */
static void
ppsdevice_shutdown(
	int unit,		/* unit number (not used) */
	struct peer *peer	/* peer structure pointer */
	)
{
	struct refclockproc *pp;
	register struct ppsunit *up;

	pp = peer->procptr;
	up = (struct ppsunit *)pp->unitptr;
	if (up->fddev > 0)
		close(up->fddev);
	if (pps_peer == peer)
		pps_peer = 0;
	free(up);
}


/*
 * ppsdevice_pps - receive data from the PPS interface
 *
 * This routine snatches the PPS timestamps from the kernel and saves the
 * sign-extended fraction in a circular buffer for processing at the
 * next poll event.
 */
static int
ppsdevice_pps(
	struct peer *peer	/* peer structure pointer */
	)
{
	register struct ppsunit *up;
	struct refclockproc *pp;
	struct ppsclockev ev;
	struct timespec timeout, ts;
	double dtemp;

	/*
	 * Convert the timespec nanoseconds field to signed double and
	 * save in the median filter. for billboards. No harm is done if
	 * previous data are overwritten. If the discipline comes bum or
	 * the data grow stale, just forget it.
	 */ 
	pp = peer->procptr;
	up = (struct ppsunit *)pp->unitptr;
	timeout.tv_sec = 0;
	timeout.tv_nsec = 0;
	memcpy(&ev, &up->ev, sizeof(struct ppsclockev));
        if( ioctl( up->fddev, CIOGETEV, (caddr_t)(&up->ev) ) < 0 ) {
	  if( debug > 0 )
            printf( "Error returned from pps driver\n" );

	  return -1;
	}
        if( ev.serial == up->ev.serial ) {
	  if( debug > 0 )
            printf( "Duplicated serial number from pps driver\n" );

	  return -1;
	}
	ts.tv_sec = up->ev.tv.tv_sec;
	ts.tv_nsec = up->ev.tv.tv_usec; /* nsec in usec field */
#ifdef DEBUG
        if( debug > 0 )
          printf( "SEC: %ld NSEC: %ld\n", ts.tv_sec, ts.tv_nsec );
#endif
	up->ts = ts;
	pp->lastrec.l_ui = ts.tv_sec + JAN_1970;
	dtemp = ts.tv_nsec * FRAC / 1e9;
	if (dtemp >= FRAC)
		pp->lastrec.l_ui++;
	pp->lastrec.l_uf = (u_int32)dtemp;
	if (ts.tv_nsec > NANOSECOND / 2)
		ts.tv_nsec -= NANOSECOND;
	dtemp = -(double)ts.tv_nsec / NANOSECOND;
	SAMPLE(dtemp + pp->fudgetime1);
#ifdef DEBUG
	if (debug > 1)
		printf("ppsdevice_pps %f %f\n", dtemp, pp->fudgetime1);
#endif

	return (0);
}

/*
 * ppsdevice_poll - called by the transmit procedure
 *
 * This routine is called once per second when in burst mode to save PPS
 * sample offsets in the median filter. At the end of the burst period
 * the samples are processed as a heap and the clock filter updated.
 */
static void
ppsdevice_poll(
	int unit,		/* unit number (not used) */
	struct peer *peer	/* peer structure pointer */
	)
{
	struct refclockproc *pp;
	int err;

	/*
	 * Accumulate samples in the median filter. If a noise sample,
	 * return with no prejudice; if a protocol error, get mean;
	 * otherwise, cool. At the end of each poll interval, do a
	 * little bookeeping and process the surviving samples.
	 */
	pp = peer->procptr;
	pp->polls++;
	err = ppsdevice_pps(peer);
	if (err < 0) {
		refclock_report(peer, CEVNT_FAULT);
		return;
	}

	/*
	 * Valid time is returned only if the prefer peer has survived
	 * the intersection algorithm and within clock_max of local time
	 * and not too long ago. This ensures the PPS time is within
	 * +-0.5 s of the local time and the seconds numbering is
	 * unambiguous. Note that the leap bits are set no-warning on
	 * the first valid update and the stratum is set at the prefer
	 * peer, unless overriden by a fudge command.
	 */
	if (peer->burst > 0)
		return;
	peer->leap = LEAP_NOTINSYNC;
	if (pp->codeproc == pp->coderecv) {
		refclock_report(peer, CEVNT_TIMEOUT);
		peer->burst = ASTAGE;
		return;

	} else if (sys_prefer == NULL) {
		pp->codeproc = pp->coderecv;
		peer->burst = ASTAGE;
		return;
	} else if (fabs(sys_prefer->offset) > clock_max) {
		pp->codeproc = pp->coderecv;
		peer->burst = ASTAGE;
		return;
	}
	pp->leap = LEAP_NOWARNING;
	if (pp->stratum >= STRATUM_UNSPEC)
		peer->stratum = sys_prefer->stratum;
	else
		peer->stratum = pp->stratum;

	if (peer->stratum == STRATUM_REFCLOCK || peer->stratum ==
	    STRATUM_UNSPEC)
		peer->refid = pp->refid;
	else
		peer->refid = addr2refid(&sys_prefer->srcadr);
	pp->lastref = pp->lastrec;
	refclock_receive(peer);
	peer->burst = ASTAGE;
}
#else
int refclock_ppsdevice_bs;
#if 0
int
pps_sample(
	   l_fp *offset		/* PPS offset */
	   )
{
	return 1;
}
#endif
#endif /* REFCLOCK */
