/*-
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer
 *    in this position and unchanged.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software withough specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#include "ppsd.h"

#if NPPSD > 0

#include "opt_ppsd.h"
#include "opt_ntp.h"
#include <sys/errno.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/conf.h>
#include <sys/kernel.h>
#include <sys/time.h>

#include <machine/clock.h>

#include <i386/isa/isa.h>
#include <i386/isa/isa_device.h>
#include <i386/isa/ppsclock.h>
#include <sys/ioccom.h>
#include <machine/cpufunc.h>
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/proc.h>

#define PPS_DEBUG 1

#define UNIT(d) minor(d)&3
#define MAXREADS	3600

/*
 * Port offsets for interrupt latency counter.
 */
#define PPS_LATENCY0	0
#define PPS_LATENCY1	1
#define PPS_LATENCY2	2

/*
 * Port offsets for pps output register.
 */
#define PPS_PPSOUT0	0
#define PPS_PPSOUT1	1
#define PPS_PPSOUT2	2

/*
 * Port offsets for microstepper register.
 */
#define PPS_MICROSTEP0	3
#define PPS_MICROSTEP1	4
#define PPS_MICROSTEP2	5

/*
 * Port offset for clock mode select and ifc to icd2061 freq. synt.
 */
#define PPS_GENPORT	6
#define PPS_EXTCLOCK	0x04		/* External/internal clock select */


/*
 * Port offsets for set/reset flipflops.
 */
#define PPS_EN_LATCH	3
#define PPS_DIS_LATCH	4
#define PPS_DIS_IRQ	5
#define PPS_EN_IRQ	6
#define PPS_RES_IRQ	7

/*
 * State bit indicating sleeping processes.
 */
#define PPS_ASLEEP	0x00000001

#ifndef PPS_HZ
#define PPS_HZ 10000000
#endif
#define MHZ_PERIOD	(1000000000/ppsd[ unit ].PpsHz)
#define INIT_STEP	(2-ppsd[ unit ].PpsHz)

static struct {
    struct ppsclockev time;
    int port;
    int nreads;
    int state;
    int microstep;
    int ppsoutput;
    int PpsHz;
    int PpsInternalClock;
    struct ppsstat stat;
    long usecs[ 3 ];
#if PPS_DEBUG
    struct ppsDebug ppsDbg;
#endif
} ppsd[NPPSD];

static int ppsdprobe (struct isa_device *);
static int ppsdattach (struct isa_device *);
static ointhand2_t ppsdintr;

struct isa_driver ppsddriver = { ppsdprobe, ppsdattach, "ppsd" };

#define CDEV_MAJOR 200
static int ppsdopen( dev_t, int, int, struct proc * );
static int ppsdclose( dev_t, int, int, struct proc * );
static int ppsdioctl( dev_t, u_long, caddr_t, int, struct proc * );

static struct cdevsw ppsd_cdevsw = 
	{ ppsdopen,	ppsdclose,	noread,		nowrite,
	  ppsdioctl,	nopoll,		nommap,		nostrategy,
	  "ppsd",	CDEV_MAJOR,	nodump,		nopsize,
	  NULL,	-1 };

static int NsToPeriods( int Ns, int PpsHz )
{
  int Res;
  long long llRes, llNs;

  llNs = (long long)(Ns) * 1000000000;
  llRes = 1000000000;
  llRes *= 1000000000;
  llRes /= (long long)(PpsHz);
  llRes = llNs / llRes;
  Res = (int)(llRes);
  return( Res );
}

static int ppsdprobe( struct isa_device *dev )
{
    return 1;
}

static int ppsdattach( struct isa_device *dev )
{
    int	unit = dev->id_unit;
    int port = dev->id_iobase;
    int InitStep;

    dev->id_intr = (inthand2_t *)ppsdintr;
    ppsd[unit].port = dev->id_iobase;
    ppsd[ unit ].nreads = 0;
    ppsd[ unit ].state = 0;
    ppsd[ unit ].stat.minhw = 999999999;
    ppsd[ unit ].stat.maxhw = 0;
    ppsd[ unit ].stat.minsys = 999999999;
    ppsd[ unit ].stat.maxsys = 0;
    ppsd[ unit ].usecs[ 0 ] = 0;
    ppsd[ unit ].usecs[ 1 ] = 0;
    ppsd[ unit ].usecs[ 2 ] = 0;
    ppsd[ unit ].PpsHz = PPS_HZ;
#ifdef PPS_INTERNALCLOCK
    ppsd[ unit ].PpsInternalClock = 0x04;
#else
    ppsd[ unit ].PpsInternalClock = 0;
#endif
#if PPS_DEBUG
    ppsd[ unit ].ppsDbg.Latency = 0;
    ppsd[ unit ].ppsDbg.nanoTime = 0L;
#endif
    inb( port + PPS_RES_IRQ );
    inb( port + PPS_EN_IRQ );
    inb( port + PPS_EN_LATCH );
    ppsd[ unit ].ppsoutput = 0;
    InitStep = 2 - ppsd[ unit ].PpsHz;
    outb( port + PPS_PPSOUT0, InitStep & 0xff );
    outb( port + PPS_PPSOUT1, ( InitStep >> 8 ) & 0xff );
    outb( port + PPS_PPSOUT2, ( InitStep >> 16 ) & 0xff );
    ppsd[ unit ].microstep = 0;
#ifdef PPS_NOMICROSTEP
    InitStep = -1;
#endif
    outb( port + PPS_MICROSTEP0, InitStep & 0xff );
    outb( port + PPS_MICROSTEP1, ( InitStep >> 8 ) & 0xff );
    outb( port + PPS_MICROSTEP2, ( InitStep >> 16 ) & 0xff );
    outb( port + PPS_GENPORT, ppsd[ unit ].PpsInternalClock | 0x03 );
    make_dev(&ppsd_cdevsw, 0, 0, 0, 0644, "ppsd");
    return( 1 );
}


static int ppsdopen( dev_t dev, int flags, int fmt, struct proc *p )
{
  return 0;
}

static int ppsdclose( dev_t dev, int flags, int fmt, struct proc *p )
{
  return 0;
}

static int ppsdioctl( dev_t dev, u_long cmd, caddr_t data,
		     int flag, struct proc *p )
{
  int unit = UNIT( dev );
  int Diff, port, Ecode;

  port = ppsd[ unit ].port;
  switch( cmd ) {

    case CIOGETDBG :
      bcopy( (char *)(&ppsd[ unit ].ppsDbg), (char *)(data),
	     sizeof ppsd[ unit ].ppsDbg );
      break;

    case CIOGETEV :
      bcopy( (char *)(&ppsd[ unit ].time), (char *)(data),
	     sizeof ppsd[ unit ].time );
      break;

    case CIOGETSTEP :
      *((int *)(data)) = ppsd[ unit ].microstep;
      break;

    case CIOSETSTEP :
      if( ( Ecode = suser( p ) ) != 0 )
	return( Ecode );

#ifndef PPS_NOMICROSTEP
      ppsd[ unit ].microstep = *((int *)(data));
      Diff = NsToPeriods( ppsd[ unit ].microstep, ppsd[ unit ].PpsHz );
      Diff -= 2;
      if( Diff > 0 )
	Diff = -Diff;
      else
	Diff = -ppsd[ unit ].PpsHz - Diff;

      outb( port + PPS_MICROSTEP0, Diff & 0xff );
      outb( port + PPS_MICROSTEP1, ( Diff >> 8 ) & 0xff );
      outb( port + PPS_MICROSTEP2, ( Diff >> 16 ) & 0xff );
#endif
      break;

    case CIOGETPPS :
      *((int *)(data)) = ppsd[ unit ].ppsoutput;
      break;

    case CIOSETPPS :
      if( ( Ecode = suser( p ) ) != 0 )
	return( Ecode );

      ppsd[ unit ].ppsoutput = *((int *)(data));
      Diff = NsToPeriods( ppsd[ unit ].ppsoutput, ppsd[ unit ].PpsHz );
      Diff--;
      if( Diff > 0 )
	Diff = -Diff;
      else
	Diff = -ppsd[ unit ].PpsHz - Diff;

      outb( port + PPS_PPSOUT0, Diff & 0xff );
      outb( port + PPS_PPSOUT1, ( Diff >> 8 ) & 0xff );
      outb( port + PPS_PPSOUT2, ( Diff >> 16 ) & 0xff );
      break;

    case CIOGETSTAT :
      bcopy( (char *)(&ppsd[ unit ].stat), (char *)(data),
	     sizeof ppsd[ unit ].stat );
      break;

    case CIOGETMED :
      /*
       * Use the median of the last 3 readings.
       */
      if( ppsd[ unit ].usecs[ 0 ] > ppsd[ unit ].usecs[ 1 ] ) {
        if( ppsd[ unit ].usecs[ 1 ] > ppsd[ unit ].usecs[ 2 ] )
	  Diff = ppsd[ unit ].usecs[ 1 ];		/* 0 1 2 */
        else if( ppsd[ unit ].usecs[ 2 ] > ppsd[ unit ].usecs[ 0 ] )
	  Diff = ppsd[ unit ].usecs[ 0 ];		/* 2 0 1 */
        else
	  Diff = ppsd[ unit ].usecs[ 2 ];		/* 0 2 1 */
      }
      else {
        if( ppsd[ unit ].usecs[ 1 ] < ppsd[ unit ].usecs[ 2 ] )
	  Diff = ppsd[ unit ].usecs[ 1 ];		/* 2 1 0 */
        else if( ppsd[ unit ].usecs[ 2 ] < ppsd[ unit ].usecs[ 0 ] )
	  Diff = ppsd[ unit ].usecs[ 0 ];		/* 1 0 2 */
        else
	  Diff = ppsd[ unit ].usecs[ 2 ];		/* 1 2 0 */
      }

      if( Diff > 500000000L )
        Diff -= 1000000000L;

      *((int *)(data)) = Diff;
      break;

    case CIOGETHZ :
      *((int *)(data)) = ppsd[ unit ].PpsHz;
      break;

    case CIOSETHZ :
      if( ( Ecode = suser( p ) ) != 0 )
	return( Ecode );

      ppsd[ unit ].PpsHz = *((int *)(data));
      break;

    case CIOWAITPPS :
      ppsd[ unit ].state |= PPS_ASLEEP;
      Ecode = tsleep( (caddr_t)(&ppsd[ unit ]),
		      PZERO | PCATCH, "ppswai", 150 );
      ppsd[ unit ].state &= ~PPS_ASLEEP;
      return( Ecode );

    case CIOGETCLK :
      if( ppsd[ unit ].PpsInternalClock != 0 )
	*((int *)(data)) = 1;
      else
	*((int *)(data)) = 0;
      break;

    case CIOSETCLK :
      if( ( Ecode = suser( p ) ) != 0 )
	return( Ecode );

      if( *((int *)(data)) != 0 )
	ppsd[ unit ].PpsInternalClock = 0x04;
      else
	ppsd[ unit ].PpsInternalClock = 0;

      outb( port + PPS_GENPORT, ppsd[ unit ].PpsInternalClock | 0x03 );
      break;


    default:
      return( ENXIO );
  }
  return( 0 );
}


/*
 * Interrupt routine.
 */
static void ppsdintr( int unit )
{
  struct timespec tvp;
  int Count;
  int port = ppsd[ unit ].port;
  long usec, Diff;

  inb( port + PPS_DIS_LATCH );
  nanotime( &tvp );
  Count = inb( port + PPS_LATENCY0 ) & 0xff;
  Count = ( ( inb( port + PPS_LATENCY1 ) & 0xff ) << 8 ) | Count;
  Count = ( ( inb( port + PPS_LATENCY2 ) & 0xff ) << 16 ) | Count;
  inb( port + PPS_EN_LATCH );
  inb( port + PPS_RES_IRQ );
#if PPS_DEBUG
  ppsd[ unit ].ppsDbg.Latency = Count;
  ppsd[ unit ].ppsDbg.nanoTime = tvp.tv_nsec;
#endif
#ifdef PPS_NOMICROSTEP
  Count += 2;
#endif
  Count *= MHZ_PERIOD;
  tvp.tv_nsec -= Count;
  if( tvp.tv_nsec < 0L ) {
    tvp.tv_nsec += 1000000000L;
    tvp.tv_sec--;
  }
  ppsd[ unit ].time.tv.tv_sec = tvp.tv_sec;
  ppsd[ unit ].time.tv.tv_usec = tvp.tv_nsec;
  ppsd[ unit ].time.serial++;

  /*
   * Update pps output registers twice a minute.
   */
  ppsd[ unit ].usecs[ 2 ] = ppsd[ unit ].usecs[ 1 ];
  ppsd[ unit ].usecs[ 1 ] = ppsd[ unit ].usecs[ 0 ];
  ppsd[ unit ].usecs[ 0 ] = ppsd[ unit ].time.tv.tv_usec;

  if( ppsd[ unit ].nreads % 30 == 0 ) {
    /*
     * Use the median of the last 3 readings.
     */
    if( ppsd[ unit ].usecs[ 0 ] > ppsd[ unit ].usecs[ 1 ] ) {
      if( ppsd[ unit ].usecs[ 1 ] > ppsd[ unit ].usecs[ 2 ] )
	Diff = ppsd[ unit ].usecs[ 1 ];		/* 0 1 2 */
      else if( ppsd[ unit ].usecs[ 2 ] > ppsd[ unit ].usecs[ 0 ] )
	Diff = ppsd[ unit ].usecs[ 0 ];		/* 2 0 1 */
      else
	Diff = ppsd[ unit ].usecs[ 2 ];		/* 0 2 1 */
    }
    else {
      if( ppsd[ unit ].usecs[ 1 ] < ppsd[ unit ].usecs[ 2 ] )
	Diff = ppsd[ unit ].usecs[ 1 ];		/* 2 1 0 */
      else if( ppsd[ unit ].usecs[ 2 ] < ppsd[ unit ].usecs[ 0 ] )
	Diff = ppsd[ unit ].usecs[ 0 ];		/* 1 0 2 */
      else
	Diff = ppsd[ unit ].usecs[ 2 ];		/* 1 2 0 */
    }


    if( Diff > 500000000L )
      Diff -= 1000000000L;

    ppsd[ unit ].ppsoutput = Diff;
    Diff = NsToPeriods( ppsd[ unit ].ppsoutput, ppsd[ unit ].PpsHz );
    Diff--;
    if( Diff > 0 )
      Diff = -Diff;
    else
      Diff = -ppsd[ unit ].PpsHz - Diff;

    outb( port + PPS_PPSOUT0, Diff & 0xff );
    outb( port + PPS_PPSOUT1, ( Diff >> 8 ) & 0xff );
    outb( port + PPS_PPSOUT2, ( Diff >> 16 ) & 0xff );
  }
  /*
   * Update statistics.
   */
  if( Count < ppsd[ unit ].stat.minhw )
    ppsd[ unit ].stat.minhw = Count;

  if( Count > ppsd[ unit ].stat.maxhw )
    ppsd[ unit ].stat.maxhw = Count;

  if( tvp.tv_nsec > 500000000L )
    usec = tvp.tv_nsec - 1000000000L;
  else
    usec = tvp.tv_nsec;

  if( abs( usec ) < abs( ppsd[ unit ].stat.minsys ) )
    ppsd[ unit ].stat.minsys = usec;

  if( abs( usec ) > abs( ppsd[ unit ].stat.maxsys ) )
    ppsd[ unit ].stat.maxsys = usec;

  if( ++ppsd[ unit ].nreads >= MAXREADS ) {
#ifdef PPS_PRINTSTATLOG
    printf( "ppsd%d: MinSys:%d MaxSys:%d MinHw:%d MaxHw:%d\n",
	    unit, ppsd[ unit ].stat.minsys, ppsd[ unit ].stat.maxsys,
	    ppsd[ unit ].stat.minhw, ppsd[ unit ].stat.maxhw );
#endif
    ppsd[ unit ].nreads = 0;
    ppsd[ unit ].stat.minhw = 999999999;
    ppsd[ unit ].stat.maxhw = 0;
    ppsd[ unit ].stat.minsys = 999999999;
    ppsd[ unit ].stat.maxsys = 0;
  }
  /*
   * Wake any processes waiting for the second pulse.
   */
  if( ppsd[ unit ].state & PPS_ASLEEP )
    wakeup( (caddr_t)(&ppsd[ unit ]) );

}
#endif /* NPPSD > 0 */
